﻿using Common;
using System;
using System.CodeDom;
using System.Collections.Generic;
using System.Drawing;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Windows.Forms;
using System.Xml;

namespace LedMapper
{
    public partial class Form1 : Form
    {
        private short[,] map;              // Массив индексов  светодиодов в двумерной матрице X,Y
        private byte[] counters;           // Массив для подсчета количества клеток с таким индексов на поле - для выявления дубликатов
        private byte[] colors;             // Массив индексов цветов клеток - для визуального разделения сегментов

        private Color[] palette;           // Массив цветов палитры - для визуального разделения сегментов
        private Button[] buttons;          // Массив кнопок палитры - для визуального разделения сегментов
        private Brush[] brushes;           // Кисти отрисовки клетов

        private string opt_file;           // Имя файла для хранения текущих настроек

        private byte  matrixWidth;          // Ширина и высота матрицы
        private byte  matrixHeight;
        private short matrixLength;         // КОл-во светодиодов в матрице - matrixWidth * matrixHeight 

        private Point pointLeftUp;         // Точка отсчета - левый верхний угол изображения, выводимого в области рисования
        private byte pnlSize;              // Размер стороны квадрата "точки" рисунка в пикселях


        private int xClick = -1;           // X клетки на mouseDown под курсором мыши
        private int yClick = -1;           // Y клетки на mouseDown под курсором мыши
        private int xCurrent = -1;         // X клетки на mouseMove под курсором мыши (без ограничением по размерам матрицы - может быть меньше нкля или больше размера матрицы)
        private int yCurrent = -1;         // Y клетки на mouseMove под курсором мыши 

        private bool isModified = false;   // Были изменения в разметке  
        private bool isMouseDown = false;
        private bool inProgress = false;
        private bool parseError = false;

        private byte corner = 0;           // угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
        private byte direction = 0;        // направление из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
        private byte zigzag = 0;           // тип матрицы: 0 - зигзаг, 1 - параллельная

        private string workFolder;         // Папка для загрузки / сохранения файлов карты
        private byte currentColor;         // Текущий цвет, используемый при разметки сегментов

        private LinkedList<Tuple<byte, byte, short[,], byte[], byte[]>> undo;  // Список состояний undo
        private readonly int undo_max_length = 50;                     // Максимально сохраняемое число состояний undo (макс длина списка undo)
        private int undo_index = 0;                                    // Указатель на позицию в списке undo - активная точка

        public Form1()
        {
            InitializeComponent();
            opt_file = Path.ChangeExtension(Application.ExecutablePath, ".opt");

            short max = -1;
            workFolder = Path.GetDirectoryName(Application.ExecutablePath);

            buttons = new Button[8] { btnColor1, btnColor2, btnColor3, btnColor4, btnColor5, btnColor6, btnColor7, btnColor8 };
            palette = new Color[8];
            brushes = new SolidBrush[8];
            for (var i=0; i< buttons.Length; i++)
            {
                palette[i] = buttons[i].BackColor;
                brushes[i] = new SolidBrush(palette[i]);
            }

            pointLeftUp = new Point();
            currentColor = 0;             // 0..7 - цвета; 8..F - те же цвета что и 0..7 - но старший бит - признак начала сегмента

            typeof(Panel).InvokeMember("DoubleBuffered", BindingFlags.SetProperty | BindingFlags.Instance | BindingFlags.NonPublic, null, pnlMapImage, new object[] { true });

            inProgress = true;

            if (File.Exists(opt_file))
            {
                try
                {
                    int l = Left, t = Top, w = Width, h = Height;
                    using (BinaryReader reader = new BinaryReader(File.Open(opt_file, FileMode.Open)))
                    {
                        // Загрузить настройки и разметку, сохраненные в предыдущем сеансе
                        edtFontWidth.Value = matrixWidth = reader.ReadByte();
                        edtFontHeight.Value = matrixHeight = reader.ReadByte();
                        matrixLength = Convert.ToInt16(matrixWidth * matrixHeight);

                        var ws = (FormWindowState)reader.ReadByte();

                        t = reader.ReadInt32();
                        l = reader.ReadInt32();
                        w = reader.ReadInt32();
                        h = reader.ReadInt32();

                        WindowState = ws;

                        corner = reader.ReadByte();
                        direction = reader.ReadByte();
                        zigzag = reader.ReadByte();
                        updateAreaRules();

                        createMap(matrixWidth, matrixHeight);

                        for (var i = 0; i < matrixWidth; i++)
                        {
                            for (var j = 0; j < matrixHeight; j++)
                            {
                                var index = reader.ReadInt16();
                                if (index > max) max = index;
                                map[i, j] = index;
                                if (index >= 0 && index < matrixLength)
                                {
                                    counters[index] += 1;
                                }
                            }
                        }

                        CheckWindowPosition(ref l, ref t, ref w, ref h);
                        StartPosition = FormStartPosition.Manual;

                        Left = l;
                        Top = t;
                        Width = w;
                        Height = h;

                        currentColor = (byte)(reader.ReadByte() & 0x0F);

                        for (var i = 0; i < matrixLength; i++)
                            colors[i] = (byte)(reader.ReadByte() & 0x0F);

                        workFolder = reader.ReadString();
                    }
                }
                catch { }
            } 
            
            if (matrixWidth == 0 || matrixHeight == 0) {
                matrixWidth = Convert.ToByte(edtFontWidth.Value);
                matrixHeight = Convert.ToByte(edtFontHeight.Value);
                matrixLength = Convert.ToInt16(matrixWidth * matrixHeight);
            }

            //                          width height map      counters colors
            undo = new LinkedList<Tuple<byte, byte, short[,], byte[],  byte[]>>();

            if (map == null)
                createMap(matrixWidth, matrixHeight);

            updatePaletteIndex();
            generateTextFromMap();

            lblCurrent.Text = "";
            lblIndex.Text = "";

            max++;
            edtNextIndex.Minimum = 0;
            edtNextIndex.Maximum = matrixLength - 1;
            edtNextIndex.Value = max >= edtNextIndex.Maximum ? 0 : max;

            tabControl.SelectedIndex = 0;

            isModified = false;
            inProgress = false;
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            pnlMapImage.Invalidate();
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            // создаем объект BinaryWriter
            try
            {
                using (var writer = new BinaryWriter(File.Open(opt_file, FileMode.Create)))
                {
                    // Записываем в файл значение каждого поля структуры
                    writer.Write(Convert.ToByte(edtFontWidth.Value));
                    writer.Write(Convert.ToByte(edtFontHeight.Value));

                    writer.Write(Convert.ToByte(WindowState));

                    if (WindowState == FormWindowState.Normal) {
                        writer.Write(Top);
                        writer.Write(Left);
                        writer.Write(Width);
                        writer.Write(Height);
                    } 
                    else
                    {
                        writer.Write(RestoreBounds.Top);
                        writer.Write(RestoreBounds.Left);
                        writer.Write(RestoreBounds.Width);
                        writer.Write(RestoreBounds.Height);
                    }

                    writer.Write(corner);
                    writer.Write(direction);
                    writer.Write(zigzag);

                    for (var i = 0; i < matrixWidth; i++)
                        for (var j = 0; j < matrixHeight; j++)
                            writer.Write(map[i, j]);

                    writer.Write(currentColor);

                    for (var i = 0; i < matrixLength; i++)
                        writer.Write(colors[i]);

                    writer.Write(workFolder);

                    writer.Flush();
                    writer.Close();
                }
            }
            catch { }
        }

        private void CheckWindowPosition(ref int x, ref int y, ref int w, ref int h)
        {
            Rectangle WorkingArea = Screen.GetWorkingArea(new Rectangle(x, y, w, h));
            if (x + w > WorkingArea.X + WorkingArea.Width) x = WorkingArea.X + WorkingArea.Width - w;
            if (y + h > WorkingArea.Y + WorkingArea.Height) y = WorkingArea.Y + WorkingArea.Height - h;
            if (x < WorkingArea.X) x = WorkingArea.X;
            if (y < WorkingArea.Y) y = WorkingArea.Y;
        }

        private void matrixSizeChanged(object sender, EventArgs e)
        {
            if (inProgress) return;

            AddToUndo();

            // Изменили размерность поля - пересоздать и перерисовать всё
            isModified = true;
            matrixWidth = Convert.ToByte(edtFontWidth.Value);
            matrixHeight = Convert.ToByte(edtFontHeight.Value);
            matrixLength = Convert.ToInt16(matrixWidth * matrixHeight);

            createMap(matrixWidth, matrixHeight);
            
            if (tabControl.SelectedIndex == 0)
                pnlMapImage.Invalidate();
            else
                generateTextFromMap();
        }

        private void btnClear_Click(object sender, EventArgs e)
        {
            AddToUndo();

            isModified = true;
            clearMap(0, 0, matrixWidth - 1, matrixHeight - 1);
            edtNextIndex.Value = 0;
            parseError = false;

            if (tabControl.SelectedIndex == 0)
                pnlMapImage.Invalidate();
            else
                generateTextFromMap();
        }

        private void pnlMapImage_Resize(object sender, EventArgs e)
        {
            pnlMapImage.Invalidate();
        }

        private void imagePaint(object sender, PaintEventArgs e)
        {
            var mx = Math.Max(matrixWidth, matrixHeight);
            var margin = mx < 100 ? 28 : 35;

            // Отрисовка текущего поля рисования
            var pnl_width = Convert.ToUInt32((pnlMapImage.ClientSize.Width - 2 * margin) / matrixWidth);
            var pnl_height = Convert.ToUInt32((pnlMapImage.ClientSize.Height - 2 * margin) / matrixHeight);
            
            if (pnl_width > 255) pnl_width = 255;
            if (pnl_height > 255) pnl_height = 255;

            // Размер "точки" изображения
            pnlSize = Convert.ToByte(Math.Min(pnl_width, pnl_height));

            // Левый верхний угол отрисовываемого изображения
            pointLeftUp.X = Convert.ToInt16((pnlMapImage.ClientSize.Width - matrixWidth * pnlSize) / 2);
            pointLeftUp.Y = Convert.ToInt16((pnlMapImage.ClientSize.Height - matrixHeight * pnlSize) / 2);

            var lineWidth = matrixWidth * pnlSize;
            var lineHeight = matrixHeight * pnlSize;

            int x1, x2, y1, y2;

            e.Graphics.Clear(Color.White);
            using (var pen = new Pen(Color.Black, 1.0F))
            using (var brush = new SolidBrush(Color.Black))
            {
                #region Решетка
                y1 = pointLeftUp.Y;
                y2 = y1 + lineHeight;
                for (var i = 0; i <= matrixWidth; i++)
                {
                    x1 = x2 = pointLeftUp.X + i * pnlSize;
                    e.Graphics.DrawLine(pen, x1, y1, x2, y2);
                }
                x1 = pointLeftUp.X;
                x2 = x1 + lineWidth;
                for (var i = 0; i <= matrixHeight; i++)
                {
                    y1 = y2 = pointLeftUp.Y + i * pnlSize;
                    e.Graphics.DrawLine(pen, x1, y1, x2, y2);
                }
                #endregion

                #region Номера столбцов (c 1)
                var fontSize = pnlSize / 3.0F * 2.0F;
                if (fontSize < 6.25) fontSize = 6.25F;
                if (fontSize > 12) fontSize = 12F;

                var stringFormat1 = new StringFormat();
                var stringFormat2 = new StringFormat();
                stringFormat1.Alignment = StringAlignment.Center;
                stringFormat1.LineAlignment = StringAlignment.Center;
                stringFormat2.Alignment = StringAlignment.Center;
                stringFormat2.LineAlignment = StringAlignment.Center;

                var font = new Font("Consolas", fontSize, FontStyle.Regular, GraphicsUnit.Point, 204);
                var textSize = e.Graphics.MeasureString(matrixWidth.ToString(), font);
                var textWidth = Convert.ToInt32(textSize.Width);
                var textHeight = Convert.ToInt32(textSize.Height);

                if (textWidth > pnlSize - 4)
                {
                    textHeight = textHeight * (matrixWidth > 99 ? 3 : 2);
                    stringFormat1.Alignment = StringAlignment.Center;
                    stringFormat1.LineAlignment = StringAlignment.Far;
                    stringFormat2.Alignment = StringAlignment.Center;
                    stringFormat2.LineAlignment = StringAlignment.Near;
                }

                y1 = pointLeftUp.Y - textHeight;
                y2 = pointLeftUp.Y + matrixHeight * pnlSize + 4;
                for (var i = 0; i < matrixWidth; i++)
                {
                    var left = pointLeftUp.X + i * pnlSize;
                    var rectUp = new Rectangle(left, y1, pnlSize, textHeight);
                    var rectDn = new Rectangle(left, y2, pnlSize, textHeight);
                    e.Graphics.DrawString((i+1).ToString(), font, Brushes.Black, rectUp, stringFormat1);
                    e.Graphics.DrawString((i+1).ToString(), font, Brushes.Black, rectDn, stringFormat2);
                }
                #endregion

                #region Номера строк (c 1)
                textSize = e.Graphics.MeasureString(matrixHeight.ToString(), font);
                textWidth = Convert.ToInt32(textSize.Width) + 4;

                stringFormat1.Alignment = StringAlignment.Center;
                stringFormat1.LineAlignment = StringAlignment.Center;
                stringFormat2.Alignment = StringAlignment.Center;
                stringFormat2.LineAlignment = StringAlignment.Center;

                x1 = pointLeftUp.X - textWidth;
                x2 = pointLeftUp.X + matrixWidth * pnlSize;
                for (var i = 0; i < matrixHeight; i++)
                {
                    var top = pointLeftUp.Y + i * pnlSize;
                    var rectLt = new Rectangle(x1, top, textWidth, pnlSize);
                    var rectRt = new Rectangle(x2, top, textWidth, pnlSize);
                    e.Graphics.DrawString((i+1).ToString(), font, Brushes.Black, rectLt, stringFormat1);
                    e.Graphics.DrawString((i+1).ToString(), font, Brushes.Black, rectRt, stringFormat2);
                }
                #endregion

                #region Отрисовка занятых клеток
                fontSize = pnlSize / 3.0F;
                if (fontSize < 3.75) fontSize = 3.75F;
                if (fontSize > 10) fontSize = 10F;
                font = new Font("Consolas", fontSize, FontStyle.Regular, GraphicsUnit.Point, 204);

                var xx1 = Math.Min(xClick, xCurrent); if (xx1 < 0) xx1 = 0; if (xx1 >= matrixWidth) xx1 = matrixWidth - 1;
                var xx2 = Math.Max(xClick, xCurrent); if (xx2 < 0) xx2 = 0; if (xx2 >= matrixWidth) xx2 = matrixWidth - 1;
                var yy1 = Math.Min(yClick, yCurrent); if (yy1 < 0) yy1 = 0; if (yy1 >= matrixHeight) yy1 = matrixHeight - 1;
                var yy2 = Math.Max(yClick, yCurrent); if (yy2 < 0) yy2 = 0; if (yy2 >= matrixHeight) yy2 = matrixHeight - 1;

                for (var x = 0; x < matrixWidth; x++)
                {
                    for (var y = 0; y < matrixHeight; y++)
                    {
                        var index = map[x,y];
                        var isDoubled = isDouble(x, y);
                        var inRange = isMouseDown && x >= xx1 && x <= xx2 && y >= yy1 && y <= yy2;

                        if (index >= 0 || inRange)
                        {
                            var fore = Brushes.White;
                            var back = isDoubled ? Brushes.Red : brushes[getCellColor(x, y)];
                            if (inRange)
                            {
                                back = index < 0 ? Brushes.LightCyan : (isDoubled ? Brushes.Salmon : Brushes.DarkCyan);
                            }                            
                            DrawCell(e.Graphics, x, y, back, fore, font);
                        }
                    }
                }
                #endregion

                #region Выбранные клетка под мышкой - xCurrent, yCurrent
                if (xCurrent >= 0 && xCurrent < matrixWidth && yCurrent >= 0 && yCurrent < matrixHeight)
                {
                    var index = map[xCurrent, yCurrent];
                    var isDoubled = isDouble(xCurrent, yCurrent);
                    var back = index < 0 ? Brushes.Cyan : (isDoubled ? Brushes.Salmon : Brushes.DarkCyan);
                    var fore = Brushes.White;
                    DrawCell(e.Graphics, xCurrent, yCurrent, back, fore, font);
                }
                #endregion
            }
        }

        private void DrawCell(Graphics g, int x, int y, Brush back, Brush fore, Font font)
        {
            var lft = pointLeftUp.X + x * pnlSize + 1;
            var top = pointLeftUp.Y + y * pnlSize + 1;

            var rect = new Rectangle(lft, top, pnlSize - 2, pnlSize - 2);
            if (isFirstIndex(x, y))
            {
                g.FillEllipse(back, rect);
                g.DrawEllipse(Pens.Black, rect);
            }
            else
            {
                g.FillRectangle(back, lft, top, pnlSize - 1, pnlSize - 1);
            }

            var index = map[x, y];
            if (index >= 0)
            {
                var stringFormat = new StringFormat();
                stringFormat.Alignment = StringAlignment.Center;
                stringFormat.LineAlignment = StringAlignment.Center;

                var str = index.ToString();

                var textSize = g.MeasureString(str, font);
                if (textSize.Width >= pnlSize - 2) str = "x";
                g.DrawString(str, font, fore, rect, stringFormat);
            }
        }

        private bool isDouble(int x, int y)
        {
            if (x >= 0 && x < matrixWidth && y >= 0 && y < matrixHeight && map[x, y] >= 0)
            {
                return counters[map[x, y]] > 1;
            }
            return false;
        }

        private byte getCellColor(int x, int y)
        {
            if (x >= 0 && x < matrixWidth && y >= 0 && y < matrixHeight && map[x, y] >= 0)
            {
                return (byte)(colors[x + matrixWidth * y] & 0x07);
            }
            return 0;
        }

        private bool isFirstIndex(int x, int y)
        {
            if (x >= 0 && x < matrixWidth && y >= 0 && y < matrixHeight && map[x, y] >= 0)
            {
                return (colors[x + matrixWidth * y] & 0x08) > 0;

            }
            return false;
        }

        private void createMap(byte width, byte height)
        {
            matrixWidth = Math.Max((byte)4, width); 
            matrixHeight = Math.Max((byte)4, height);

            if (matrixWidth > 127) matrixWidth = 127;
            if (matrixHeight > 127) matrixHeight = 127;
            
            matrixLength = Convert.ToInt16(matrixWidth * matrixHeight);

            map = new short[width, height];
            counters = new byte[width * height];
            colors   = new byte[width * height];
            clearMap(0, 0, width - 1, height - 1);

            edtNextIndex.Value = 0;
            edtNextIndex.Minimum = 0;
            edtNextIndex.Maximum = matrixHeight * matrixWidth - 1;
        }

        private void clearMap(int x1, int y1, int x2, int y2)
        {
            for (var i = x1; i <= x2; i++)
            {
                for (var j = y1; j <= y2; j++)
                {
                    var index = map[i, j];
                    map[i, j] = -1;
                    if (index >= 0)
                    {
                        var cnt = counters[index];
                        if (cnt > 0) counters[index] -= 1;
                        colors[i + j * matrixWidth] = 0;
                    }
                }
            }
        }

        private void parseText()
        {
            var text = edtHexArray.Text.Replace('\n', ' ').Replace('\r', ' ').Replace('\t', ' ').Trim();

            /* !!! Не удалять. Это информация о распределении сегментов матрицы !!!
               >22222222111111112222222211111111222222221111111122222222111111112222222211111111222222221111111122222222111111112222222A11111119
                77777777000000007777777700000000777777770000000077777777000000007777777700000000777777770000000077777777000000007777777F00000008<
            */
            var idx1 = text.IndexOf('>');
            var idx2 = text.IndexOf('<');
            var segm = idx1 >=0 && idx2 >= 0 ? text.Substring(idx1 + 1, idx2 - idx1 - 1).Replace(" ", "").Trim().ToUpper() : "";

            // const int16_t PROGMEM map[] = { ... }
            idx1 = text.IndexOf('{');
            idx2 = text.IndexOf('}');
            text = idx1 >= 0 && idx2 >= 0 ? text.Substring(idx1 + 1, idx2 - idx1 - 1).Trim() : "";

            clearMap(0, 0, matrixWidth - 1, matrixHeight - 1);
            parseError = false;

            if (text.Length > 0)
            {
                var parts = text.Split(',');
                var i = 0;
                var len = Math.Min(matrixLength, parts.Length);

                while (i < len)
                {
                    var x = i % matrixWidth;
                    var y = i / matrixWidth;

                    short index = -1;
                    var str = parts[i].Trim();
                    var ok = short.TryParse(str, out index);
                    
                    if (!ok)
                    {
                        if (str.ToLower().StartsWith("0x")) str = str.Substring(2);
                        ok = short.TryParse(str, NumberStyles.HexNumber, CultureInfo.InvariantCulture, out index);
                    }
                    if (!ok)
                    {
                        parseError = true;
                        index = -1;                    
                    }
                    
                    if (index >= 0 && index < matrixLength)
                    {
                        map[x, y] = index;
                        if (index >= 0) counters[index] += 1;
                    }
                    else
                    {
                        parseError = true;
                    }
                    i++;
                }
            }

            if (segm.Length > 0)
            {
                for (var ix = 0; ix < segm.Length; ix++)
                {
                    char ch = segm[ix];
                    byte val = 0;
                    if (ch >= '0' && ch <= '9')
                    {
                        val = (byte)(((byte)ch - (byte)'0') & 0x0F);
                    } 
                    else if (ch >= 'A' && ch <= 'F')
                    {
                        val = (byte)((((byte)ch - (byte)'A') & 0x0F) + 10);
                    }
                    colors[ix] = val;
                }
            }

            if (!parseError) tabControl.SelectedIndex = 0;

            SetFormState();
        }

        private void updatePaletteIndex()
        {
            foreach (var button in buttons) { button.Text = ""; };
            buttons[currentColor].Text = "X";
        }

        private void generateTextFromMap()
        {
            var num_len = matrixLength.ToString().Length;
            var cnt = 0;
            var brk = matrixWidth <= 16 ? 64 : 128; 

            // Здесь не может использоваться uint8_t, поскольку для матриц размером 256 все равно кроме индекса 0..255
            // может быть значение -1 - индекс не используется (нет соответствия позиции x,y  номеру светодиода, светодиод не зажигать)
            // Это может обеспечить только применение знакового int16_t
            var format = rbHex.Checked ? "0x{0:X4}" : $"{{0,{num_len}}}";
            var sb = new StringBuilder();
            sb.Append("/* !!! Не удалять. Это информация о распределении сегментов матрицы !!!\r\n>");
            for (var i = 0; i < matrixLength; i++)
            {
                sb.Append(string.Format("{0:X1}", colors[i] & 0x0F));
                cnt++;
                if (cnt == brk && i < matrixLength - 1)
                {
                    cnt = 0;
                    sb.Append("\r\n ");
                }
            }
            sb.Append("<\r\n*/\r\n");

            cnt = 0;
            sb.Append($"const int16_t PROGMEM imap[] = {{\r\n  ");
            for (var y = 0; y < matrixHeight; y++)
            {
                for (var x = 0; x < matrixWidth; x++)
                {
                    sb.Append(string.Format(format, map[x, y]));
                    if (cnt < matrixLength - 1) sb.Append(", ");
                    cnt++;
                }
                sb.Append("\r\n");
                if (y + 1 < matrixHeight) sb.Append("  ");
            }
            sb.Append("};\r\n");
            edtHexArray.Text = sb.ToString();

            parseError = false;
            isModified = false;
        }

        private void tabControl_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (isModified) generateTextFromMap();
            SetFormState();
        }

        private void pnlCharImage_MouseDown(object sender, MouseEventArgs e)
        {
            var x = e.Location.X - pointLeftUp.X;
            var y = e.Location.Y - pointLeftUp.Y;
            if (x < 0 || y < 0 || x > matrixWidth * pnlSize || y > matrixHeight * pnlSize) return;

            isMouseDown = true;

            xClick = Convert.ToByte(Math.Floor(1.0 * x / pnlSize));
            yClick = Convert.ToByte(Math.Floor(1.0 * y / pnlSize));

            pnlMapImage.Invalidate();
        }

        private void pnlMapImage_MouseUp(object sender, MouseEventArgs e)
        {
            if (!isMouseDown) return;

            pnlShowSize.Visible = false;

            var x = e.Location.X - pointLeftUp.X;
            var y = e.Location.Y - pointLeftUp.Y;

            xCurrent = Convert.ToByte(Math.Floor(1.0 * x / pnlSize));
            yCurrent = Convert.ToByte(Math.Floor(1.0 * y / pnlSize));

            if (xClick < 0) xClick = 0; if (xClick >= matrixWidth) xClick = matrixWidth - 1;
            if (yClick < 0) yClick = 0; if (yClick >= matrixHeight) yClick = matrixHeight - 1;
            if (xCurrent < 0) xCurrent = 0; if (xCurrent >= matrixWidth) xCurrent = matrixWidth - 1;
            if (yCurrent < 0) yCurrent = 0; if (yCurrent >= matrixHeight) yCurrent = matrixHeight - 1;

            // Размер области - одна клетка?
            if (xClick == xCurrent && yClick == yCurrent)
            {
                var idx = map[xClick, yClick]; 
                if (e.Button == MouseButtons.Left)
                {
                    if (idx < 0)
                    {
                        // Клик левой кнопкой мыши и поле еще не заполнено - поместить в поле следующий индекс
                        AddToUndo();
                        idx = Convert.ToInt16(edtNextIndex.Value);
                        map[xClick, yClick] = idx;
                        counters[idx] += 1;
                        colors[xClick + yClick * matrixWidth] = currentColor;
                        if (idx < edtNextIndex.Maximum)
                        {
                            edtNextIndex.Value = idx + 1;
                        }
                    }
                    else
                    {
                        // Клик левой кнопкой мыши и поле заполнено - редактировать ячейку
                        editCell((short)xClick, (short)yClick);
                    }
                    isModified = true;
                } 
                else if (e.Button == MouseButtons.Right && idx >= 0)
                {
                    // Клик правой кнопкой мыши - сбросить поле
                    AddToUndo();
                    map[xClick, yClick] = -1;
                    counters[idx] -= 1;
                    colors[xClick + yClick * matrixWidth] = 0;
                    isModified = true;
                }                
            }
            else
            {
                // Левый верхний и правый нижний угол выделенной области
                var x1 = Convert.ToByte(Math.Min(xClick, xCurrent));
                var x2 = Convert.ToByte(Math.Max(xClick, xCurrent));
                var y1 = Convert.ToByte(Math.Min(yClick, yCurrent));
                var y2 = Convert.ToByte(Math.Max(yClick, yCurrent));

                if (e.Button == MouseButtons.Right)
                {
                    // Очистить область, попадающие в координаты x1,y1 .. x2,y2
                    AddToUndo();
                    clearMap(x1, y1, x2, y2);
                    isModified = true;
                }
                else
                {
                    AddToUndo();
                    var index = Convert.ToInt16(edtNextIndex.Value);
                    var flag = false;
                    var first = true;

                    // Заполнить выделенную область в соответствии с текущим выбранным правилом заполнения номеров:
                    // corner     угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
                    // direction  направление из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
                    // zigzag     тип матрицы: 0 - зигзаг, 1 - параллельная

                    switch (corner)
                    {
                        case 0:       // левый нижний   - направление - вверх или вправо
                            if (direction == 1) // вверх, потом вправо
                            {
                                for (short xx = x1; xx <= x2; xx++)
                                {
                                    if (zigzag != 0 || zigzag == 0 && !flag)
                                        for (short yy = y2; yy >= y1; yy--) { setIndex(xx, yy, index++, first); first = false; }
                                    else
                                        for (short yy = y1; yy <= y2; yy++) { setIndex(xx, yy, index++, first); first = false; }
                                    flag = !flag;
                                }
                            }
                            else
                            if (direction == 0) // вправо, потом вверх
                            {
                                for (short yy = y2; yy >= y1; yy--)
                                {
                                    if (zigzag != 0 || zigzag == 0 && !flag)
                                        for (short xx = x1; xx <= x2; xx++) { setIndex(xx, yy, index++, first); first = false; }
                                    else
                                        for (short xx = x2; xx >= x1; xx--) { setIndex(xx, yy, index++, first); first = false; }
                                    flag = !flag;
                                }
                            }
                            break;
                        case 1:       // левый верхний  - направление - вниз или вправо
                            if (direction == 3) // вниз, потом вправо
                            {
                                for (short xx = x1; xx <= x2; xx++)
                                {
                                    if (zigzag != 0 || zigzag == 0 && !flag)
                                        for (short yy = y1; yy <= y2; yy++) { setIndex(xx, yy, index++, first); first = false; }
                                    else
                                        for (short yy = y2; yy >= y1; yy--) { setIndex(xx, yy, index++, first); first = false; }
                                    flag = !flag;
                                }
                            }
                            else
                            if (direction == 0) // вправо, потом вниз
                            {
                                for (short yy = y1; yy <= y2; yy++)
                                {
                                    if (zigzag != 0 || zigzag == 0 && !flag)
                                        for (short xx = x1; xx <= x2; xx++) { setIndex(xx, yy, index++, first); first = false; }
                                    else
                                        for (short xx = x2; xx >= x1; xx--) { setIndex(xx, yy, index++, first); first = false; }
                                    flag = !flag;
                                }
                            }
                            break;
                        case 2:       // правый верхний - направление - вниз или влево
                            if (direction == 3) // вниз, потом влево
                            {
                                for (short xx = x2; xx >= x1; xx--)
                                {
                                    if (zigzag != 0 || zigzag == 0 && !flag)
                                        for (short yy = y1; yy <= y2; yy++) { setIndex(xx, yy, index++, first); first = false; }
                                    else
                                        for (short yy = y2; yy >= y1; yy--) { setIndex(xx, yy, index++, first); first = false; }
                                    flag = !flag;
                                }
                            }
                            else
                            if (direction == 2) // влево, потом вниз
                            {
                                for (short yy = y1; yy <= y2; yy++)
                                {
                                    if (zigzag != 0 || zigzag == 0 && !flag)
                                        for (short xx = x2; xx >= x1; xx--) { setIndex(xx, yy, index++, first); first = false; }
                                    else
                                        for (short xx = x1; xx <= x2; xx++) { setIndex(xx, yy, index++, first); first = false; }
                                    flag = !flag;
                                }
                            }
                            break;
                        case 3:       // правый нижний - направление - вверх или влево
                            if (direction == 1) // вверх, потом влево
                            {
                                for (short xx = x2; xx >= x1; xx--)
                                {
                                    if (zigzag != 0 || zigzag == 0 && !flag)
                                        for (short yy = y2; yy >= y1; yy--) { setIndex(xx, yy, index++, first); first = false; }
                                    else
                                        for (short yy = y1; yy <= y2; yy++) { setIndex(xx, yy, index++, first); first = false; }
                                    flag = !flag;
                                }
                            }
                            else
                            if (direction == 2) // влево, потом вверх
                            {
                                for (short yy = y2; yy >= y1; yy--)
                                {
                                    if (zigzag != 0 || zigzag == 0 && !flag)
                                        for (short xx = x2; xx >= x1; xx--) { setIndex(xx, yy, index++, first); first = false; }
                                    else
                                        for (short xx = x1; xx <= x2; xx++) { setIndex(xx, yy, index++, first); first = false; }
                                    flag = !flag;
                                }
                            }
                            break;
                    }
                    edtNextIndex.Value = index < edtNextIndex.Maximum ? index : 0;
                    isModified = true;
                }
            }

            xClick = -1;
            yClick = -1;

            isMouseDown = false;
            pnlMapImage.Invalidate();
        }

        private void pnlMapImage_MouseMove(object sender, MouseEventArgs e)
        {
            if (pnlSize > 0) { 
                var x = e.Location.X - pointLeftUp.X;
                var y = e.Location.Y - pointLeftUp.Y;

                var xc = Convert.ToInt32(Math.Floor(1.0 * x / pnlSize));
                var yc = Convert.ToInt32(Math.Floor(1.0 * y / pnlSize));

                if (xc != xCurrent || yCurrent != yc)
                {
                    xCurrent = xc;
                    yCurrent = yc;
                    toolTip1.Hide(pnlMapImage);
                }

                if (xc >= 0 && xc < matrixWidth && yc >= 0 && yc < matrixHeight)
                {
                    var index = map[xc, yc];
                    lblCurrent.Text = $"[ {xc + 1}, {yc + 1} ]";
                    lblIndex.Text = index >= 0 ? $"{index}" : "";
                }
                else
                {
                    lblCurrent.Text = "";
                    lblIndex.Text = "";
                }

                if (isMouseDown)
                {
                    var x1 = Convert.ToInt16(Math.Min(xClick, xCurrent));
                    var x2 = Convert.ToInt16(Math.Max(xClick, xCurrent));
                    var y1 = Convert.ToInt16(Math.Min(yClick, yCurrent));
                    var y2 = Convert.ToInt16(Math.Max(yClick, yCurrent));
                    var ww = x2 - x1 + 1;
                    var hh = y2 - y1 + 1;
                    if ((ww > 1 || hh > 1) && xc >= 0 && xc < matrixWidth && yc >= 0 && yc < matrixHeight)
                    {
                        pnlShowSize.Location = new Point(e.X + 24, e.Y + 30);
                        lblShowSize.Text = $"{ww} x {hh}";
                        pnlShowSize.Visible = true;
                    } 
                    else
                    {
                        pnlShowSize.Visible = false;
                    }
                }
                else
                {
                    pnlShowSize.Visible = false;
                }
            } 
            else
            {
                lblCurrent.Text = "";
            }

            pnlMapImage.Invalidate();
        }

        private void toolTip1_Popup(object sender, PopupEventArgs e)
        {
            if (inProgress) return;

            if (e.AssociatedControl == pnlMapImage)
            {
                if (xCurrent >= 0 && xCurrent < matrixWidth && yCurrent >= 0 && yCurrent < matrixHeight)
                {
                    var index = map[xCurrent, yCurrent];
                    if (index >= 0)
                    {
                        inProgress = true;
                        toolTip1.SetToolTip(pnlMapImage, index.ToString());
                        inProgress = false;
                    } 
                    else
                    {
                        e.Cancel = true;
                    }
                }
                else
                {
                    e.Cancel = true;
                }

            }
        }

        private void clearIndex(short x, short y)
        {
            var idx = map[x, y];
            if (idx >= 0)
            {
                map[x, y] = -1;
                counters[idx] -= 1;
                colors[x + matrixWidth * y] = 0;
            }
        }

        private void setIndex(short x, short y, short index, bool first)
        {
            // x, y - short, потому что циклы, откуда вызывается эта функция
            // бывают с обратным отсчетом - вроде while(...; x >= 0; x--)
            // При использовании byte конструкция "x--", при значении x=0 вернет значение x=255 -
            // условие "x >= 0" не выполнится, в функцию передастся 255 - выход за границы массива и exception
            if (index >= matrixLength) return;

            if (map[x, y] < 0)
            {
                map[x, y] = index;
                if (index >= 0) counters[index] += 1;
            } 
            else
            {
                var oldIndex = map[x, y];
                counters[oldIndex] -= 1;
                map[x, y] = index;
                if (index >= 0) counters[index] += 1;
            }
            colors[x + matrixWidth * y] = first ? (byte)((currentColor & 0x07) | 0x08) : (byte)(currentColor & 0x07);
        }

        private void SetFormState()
        {
            var hasError = counters.Any(p => p > 1);
            lblWarning.Visible = hasError;
            lblWarning2.Visible = parseError;
            btnSave.Enabled = !hasError;

            var text = edtHexArray.Text.Replace('\n', ' ').Replace('\r', ' ').Replace('\t', ' ').Trim();
            btnParse.Enabled = text.Length > 0;

            btnUndo.Enabled = undo_index > 0;
            btnRedo.Enabled = undo_index < undo.Count - 1;
        }

        private void SaveToFile()
        {
            var file = Path.Combine(workFolder, $"{matrixWidth}x{matrixHeight}.map");
            saveFileDialog.InitialDirectory = workFolder;
            saveFileDialog.FileName = file;
            saveFileDialog.Filter = "Карта индексов (*.map)|*.map|Все файлы (*.*)|*.*";
            saveFileDialog.FilterIndex = 0;

            var result = saveFileDialog.ShowDialog();

            if (result != DialogResult.OK) return;

            file = saveFileDialog.FileName;
            workFolder = Path.GetDirectoryName(file);

            // создаем объект BinaryWriter
            try
            {
                using (var writer = new BinaryWriter(File.Open(file, FileMode.Create)))
                {
                    // Записываем в файл значение каждого поля структуры
                    writer.Write((ushort)0xA5A5);
                    writer.Write(Convert.ToByte(matrixWidth));
                    writer.Write(Convert.ToByte(matrixHeight));
                    
                    // Карта индексов
                    for (var y = 0; y < matrixHeight; y++)
                        for (var x = 0; x < matrixWidth; x++)
                            writer.Write(map[x, y]);

                    // Карта цветов распределения по сегментам (при загрузке в прошивке она будет игнорироваться)
                    // Цвет - 0..7, если установлен старший вит 4-х битной комбинации - 8..FF - это признак начала сегмента
                    // Один цвет - 4 бита, значит в байт - два цвета
                    for (var i = 0; i < matrixLength - 1; i+=2)
                    {
                        writer.Write((byte)((colors[i] << 4) | (colors[i + 1])));
                    }

                    writer.Flush();
                    writer.Close();
                }

                using (new CenterWinDialog(this))
                {
                    MessageBox.Show($"Карта индексов сохранена в файл:\n\n   {file}", "Информация", MessageBoxButtons.OK, MessageBoxIcon.Information);
                }
            }
            catch (Exception ex)
            {
                using (new CenterWinDialog(this))
                {
                    MessageBox.Show($"Ошибка сохранения карты индексов:\n\n   {ex.Message}", "Ошибка", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
        }

        private void LoadFromFile()
        {
            openFileDialog.InitialDirectory = workFolder;
            openFileDialog.Filter = "Карта индексов (*.map)|*.map|Все файлы (*.*)|*.*";
            openFileDialog.FilterIndex = 0;

            var result = openFileDialog.ShowDialog();

            if (result == DialogResult.OK && File.Exists(openFileDialog.FileName))
            {
                var file = openFileDialog.FileName;
                workFolder = Path.GetDirectoryName(file);

                try
                {
                    int l = Left, t = Top, w = Width, h = Height;
                    using (BinaryReader reader = new BinaryReader(File.Open(file, FileMode.Open)))
                    {
                        // Загрузить настройки и разметку, сохраненные в предыдущем сеансе
                        var sign = reader.ReadUInt16();
                        if (sign != 0xA5A5) throw new Exception("Неверный формат файла карты индексов");

                        var ww = reader.ReadByte();
                        var hh = reader.ReadByte();
                        if (ww > 127 || hh > 127) throw new Exception("Карта индексов имеет неподдерживаемые размеры");

                        edtFontWidth.Value = matrixWidth = ww;
                        edtFontHeight.Value = matrixHeight = hh;

                        createMap(matrixWidth, matrixHeight);

                        for (var y = 0; y < matrixHeight; y++)
                        {
                            for (var x = 0; x < matrixWidth; x++)
                            {
                                var index = reader.ReadInt16();
                                map[x, y] = index;
                                if (index >= 0 && index < matrixLength)
                                {
                                    counters[index] += 1;
                                }
                            }
                        }

                        // Восстановить карту цветов распределения по сегментам (при загрузке в прошивке она будет игнорироваться)
                        // Карта цветов распределения по сегментам (при загрузке в прошивке она будет игнорироваться)
                        // Цвет - 0..7, если установлен старший вит 4-х битной комбинации - 8..FF - это признак начала сегмента
                        // Один цвет - 4 бита, значит в байт - два цвета
                        for (var i = 0; i < matrixLength - 1; i += 2)
                        {
                            byte val = reader.ReadByte();
                            colors[i] = (byte)((val & 0xF0) >> 4);
                            colors[i+1] = (byte)(val & 0x0F);
                        }
                    }

                    pnlMapImage.Invalidate();
                    Application.DoEvents();

                    using (new CenterWinDialog(this))
                    {
                        MessageBox.Show($"Карта индексов загружена из файла:\n\n   {file}", "Информация", MessageBoxButtons.OK, MessageBoxIcon.Information);
                    }

                }
                catch (Exception ex)
                {
                    using (new CenterWinDialog(this))
                    {
                        MessageBox.Show($"Ошибка загрузки карты индексов:\n\n   {ex.Message}", "Ошибка", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }
            }
        }    

        #region Правила заполнения области

        // corner     угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
        // direction  направление из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
        //            ImageList: 0,1 - вниз; 2,3 - влево; 4,5 - вправо; 6,7 - вверх; 0,2,4,6 - синий; 1,3,5,7 - серый
        // zigzag     тип матрицы: 0 - зигзаг, 1 - параллельная
        //            ImageList2:
        //              0 - параллель снизу вверх
        //              1 - параллель слева направо
        //              2 - параллель справа налево
        //              3 - параллель сверху вниз
        //              4 - зигзаг горизонтальный
        //              5 - зигзаг вертикальный

        private void updateAreaRules()
        {
            inProgress = true;

            // Отмеченный чекбокс
            chkLeftTop.Checked = corner == 1;
            chkRightTop.Checked = corner == 2;
            chkLeftDown.Checked = corner == 0;
            chkRightDown.Checked = corner == 3;

            // Сбросить все стрелки направлений к серому
            btnDownToLeft.ImageIndex = 3;
            btnDownToRight.ImageIndex = 5;
            btnTopToLeft .ImageIndex = 3;
            btnTopToRight.ImageIndex = 5;
            btnLeftToDown.ImageIndex = 1;
            btnLeftToUp.ImageIndex = 7;
            btnRightToDown.ImageIndex = 1;
            btnRightToUp.ImageIndex = 7;

            // Отмеченная стрелка в зависимости от угла и направления
            switch (corner)
            {
                case 0:     // левый нижний
                    if (direction == 0) /* вправо */ btnDownToRight.ImageIndex = 4;
                    if (direction == 1) /* вверх  */ btnLeftToUp.ImageIndex = 6;
                    break;
                case 1:     // левый верхний
                    if (direction == 0) /* вправо */ btnTopToRight.ImageIndex = 4;
                    if (direction == 3) /* вниз   */ btnLeftToDown.ImageIndex = 0;
                    break;
                case 2:     // правый верхний
                    if (direction == 2) /* влево  */ btnTopToLeft.ImageIndex = 2; 
                    if (direction == 3) /* вниз   */ btnRightToDown.ImageIndex = 0;
                    break;
                case 3:     // правый нижний
                    if (direction == 2) /* влево  */ btnDownToLeft.ImageIndex = 2;
                    if (direction == 1) /* вниз   */ btnRightToUp.ImageIndex = 6;
                    break;
            }

            // Картинка выбора типа (и направления)
            if (zigzag == 0)
            {
                // Тип подключения - зигзаг
                if (direction == 0 /* вправо */ || direction == 2 /* влево */)
                {
                    // Горизонтальный зигзаг
                    btnZigZag.ImageIndex = 4;
                }
                else
                {
                    // Вертикальный зигзаг
                    btnZigZag.ImageIndex = 5;
                }
            }
            else
            {
                // Тип подключения - параллель
                switch (direction)
                {
                    case 0:     // вправо
                        btnZigZag.ImageIndex = 1;
                        break;
                    case 1:     // вверх
                        btnZigZag.ImageIndex = 0;
                        break;
                    case 2:     // влево
                        btnZigZag.ImageIndex = 2;
                        break;
                    case 3:     // вниз
                        btnZigZag.ImageIndex = 3;
                        break;
                }
            }

            inProgress = false;
        }

        // corner     угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
        // direction  направление из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
        //            ImageList: 0,1 - вниз; 2,3 - влево; 4,5 - вправо; 6,7 - вверх; 0,2,4,6 - синий; 1,3,5,7 - серый
        // zigzag     тип матрицы: 0 - зигзаг, 1 - параллельная
        //            ImageList2:
        //              0 - параллель снизу вверх
        //              1 - параллель слева направо
        //              2 - параллель справа налево
        //              3 - параллель сверху вниз
        //              4 - зигзаг горизонтальный
        //              5 - зигзаг вертикальный

        private void chkLeftTop_CheckedChanged(object sender, EventArgs e)
        {
            if (inProgress) return;
            corner = 1;   // левый верхний -- только вниз (3) или вправо (0)
            if (!(direction == 3 || direction == 0))
            {
                direction = 0;                 
            }
            updateAreaRules();
        }

        private void chkRightTop_CheckedChanged(object sender, EventArgs e)
        {
            if (inProgress) return;
            corner = 2;   // правый верхний -- только вниз (3) или влево (2)
            if (!(direction == 3 || direction == 2))
            {
                direction = 2;
            }
            updateAreaRules();
        }

        private void chkLeftDown_CheckedChanged(object sender, EventArgs e)
        {
            if (inProgress) return;
            corner = 0;   // левый нижний -- только вверх (1) или вправо (0)
            if (!(direction == 1 || direction == 0))
            {
                direction = 0;
            }
            updateAreaRules();
        }

        private void chkRightDown_CheckedChanged(object sender, EventArgs e)
        {
            if (inProgress) return;
            corner = 3;   // правый нижний -- только вверж (1) или вkево (2)
            if (!(direction == 1 || direction == 2))
            {
                direction = 2;
            }
            updateAreaRules();
        }

        // corner     угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
        // direction  направление из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
        //            ImageList: 0,1 - вниз; 2,3 - влево; 4,5 - вправо; 6,7 - вверх; 0,2,4,6 - синий; 1,3,5,7 - серый
        // zigzag     тип матрицы: 0 - зигзаг, 1 - параллельная
        //            ImageList2:
        //              0 - параллель снизу вверх
        //              1 - параллель слева направо
        //              2 - параллель справа налево
        //              3 - параллель сверху вниз
        //              4 - зигзаг горизонтальный
        //              5 - зигзаг вертикальный

        private void btnTopToRight_Click(object sender, EventArgs e)
        {
            direction = 0;
            corner = 1;
            updateAreaRules();
        }

        private void btnTopToLeft_Click(object sender, EventArgs e)
        {
            direction = 2;
            corner = 2;
            updateAreaRules();
        }

        private void btnRightToDown_Click(object sender, EventArgs e)
        {
            direction = 3;
            corner = 2;
            updateAreaRules();
        }

        private void btnRightToUp_Click(object sender, EventArgs e)
        {
            direction = 1;
            corner = 3;
            updateAreaRules();
        }

        private void btnDownToLeft_Click(object sender, EventArgs e)
        {
            direction = 2;
            corner = 3;
            updateAreaRules();
        }

        private void btnDownToRight_Click(object sender, EventArgs e)
        {
            direction = 0;
            corner = 0;
            updateAreaRules();
        }

        private void btnLeftToUp_Click(object sender, EventArgs e)
        {
            direction = 1;
            corner = 0;
            updateAreaRules();
        }

        private void btnLeftToDown_Click(object sender, EventArgs e)
        {
            direction = 3;
            corner = 1;
            updateAreaRules();
        }

        private void btnZigZag_Click(object sender, EventArgs e)
        {
            zigzag = (zigzag == 0) ? (byte)1 : (byte)0;
            updateAreaRules();
        }

        #endregion

        private void edtHexArray_TextChanged(object sender, EventArgs e)
        {
            SetFormState();
        }

        private void btnParse_Click(object sender, EventArgs e)
        {
            AddToUndo();
            parseText();
            SetFormState();
        }

        private void HexDec_CheckedChanged(object sender, EventArgs e)
        {
            generateTextFromMap();
            SetFormState();
        }

        private void btnOpen_Click(object sender, EventArgs e)
        {
            AddToUndo();
            LoadFromFile();
            generateTextFromMap();
            SetFormState();
        }

        private void btnSave_Click(object sender, EventArgs e)
        {
            SaveToFile();
            SetFormState();
        }

        private void btnRestore_Click(object sender, EventArgs e)
        {
            generateTextFromMap();
            SetFormState();
        }

        /// <summary>
        /// СОхранить текущее состояние в список Undo
        /// </summary>
        private void AddToUndo()
        {
            while (undo_index < undo.Count) undo.RemoveLast();

            var smap = map.Clone() as short[,];
            var scounters = counters.Clone() as byte[];
            var scolors = colors.Clone() as byte[];

            var node = new Tuple<byte, byte, short[,], byte[], byte[]>(matrixWidth, matrixHeight, smap, scounters, scolors);
            undo.AddLast(node);

            while (undo.Count > undo_max_length) undo.RemoveFirst();
            undo_index = undo.Count;

            SetFormState();
        }

        /// <summary>
        /// Откат состояния на 1 шаг назад
        /// </summary>
        private void Undo()
        {
            if (undo_index == 0) return;

            undo_index--;
            var index = undo.Count;
            var node = undo.Last;
            while (index - 1 > undo_index)
            {
                node = node.Previous;
                index--;
            }
            
            ApplyItem(node.Value);
            SetFormState();

            if (tabControl.SelectedIndex == 0)
                pnlMapImage.Invalidate();
            else
                generateTextFromMap();
        }

        /// <summary>
        /// Движение по списке undo на 1 шаг вперед
        /// </summary>
        private void Redo()
        {
            if (undo_index == undo.Count) return;

            undo_index++;
            var index = undo.Count;
            var node = undo.Last;
            while (index - 1 > undo_index)
            {
                node = node.Previous;
                index--;
            }

            ApplyItem(node.Value);
            SetFormState();

            if (tabControl.SelectedIndex == 0)
                pnlMapImage.Invalidate();
            else
                generateTextFromMap();
        }

        private void ApplyItem(Tuple<byte, byte, short[,], byte[], byte[]> tpl)
        {
            inProgress = true;

            var mw = tpl.Item1;
            var mh = tpl.Item2;
            var smap = tpl.Item3;
            var scounters = tpl.Item4;
            var scolors = tpl.Item5;

            if (matrixWidth != mw)
                edtFontWidth.Value = matrixWidth = mw;
            if (matrixHeight != mh)
                edtFontHeight.Value = matrixHeight = mh;

            matrixLength = Convert.ToInt16(matrixWidth * matrixHeight);

            map = smap.Clone() as short[,];
            counters = scounters.Clone() as byte[];
            colors = scolors.Clone() as byte[];

            inProgress = false;
        }

        private void btnUndo_Click(object sender, EventArgs e)
        {
            if (undo_index == undo.Count) { 
                AddToUndo();
                undo_index = undo.Count - 1;
            }
            Undo();
        }

        private void btnRedo_Click(object sender, EventArgs e)
        {
            Redo();
        }

        private void btnColor_Click(object sender, EventArgs e)
        {
            var button = sender as Button;
            if (button != null)
            {
                currentColor = Convert.ToByte(button.Tag);
                updatePaletteIndex();
            }
        }

        void editCell(short x, short y)
        {
            if (x < 0 || x >= matrixWidth || y < 0 || y >= matrixHeight) return;

            var cellIndex = map[x, y];
            var colorIndex = colors[x + matrixWidth * y];
            var cellIndexMax = Convert.ToInt16(edtNextIndex.Maximum);

            var form = new EditForm(x, y, colorIndex, cellIndex, cellIndexMax);
            form.StartPosition = FormStartPosition.Manual;

            var xPos = pointLeftUp.X + x * pnlSize + 1;
            var yPos = pointLeftUp.Y + (y + 1) * pnlSize + 1;

            form.Location = pnlMapImage.PointToScreen(new Point(xPos, yPos));

            if (form.ShowDialog() == DialogResult.OK)
            {
                if (cellIndex != form.cellIndex)
                {
                    counters[cellIndex] -= 1;
                    counters[form.cellIndex] += 1;
                    map[x, y] = form.cellIndex;
                }
                
                colors[x + matrixWidth * y] = form.colorIndex;

                isModified = true;
                pnlMapImage.Invalidate();
            }
        }

        private void Form1_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode== Keys.Z && e.Control && !e.Alt && !e.Shift && btnUndo.Enabled) {
                btnUndo.PerformClick();
            }
        }
    }
}
