﻿using System;
using System.Drawing;
using System.IO;
using System.Windows.Forms;
using Timer = System.Threading.Timer;
using System.Threading;
using System.Collections.Generic;
using System.Globalization;
using System.Text;
using System.Text.RegularExpressions;
using Common;
using Cyotek.Windows.Forms;
using JinxFramer.Properties;


// This form used Color Management components from following project: https://github.com/cyotek/Cyotek.Windows.Forms.ColorPicker

namespace JinxFramer
{
    public enum FileType
    {
        MOVIE,
        PICTURE
    };

    public partial class frmMain : Form
    {
        private const int margin = 12;

        private string opt_file;           // для хранения текущих настроек
        private string lng_file;           // для хранения настроек выбранного языка
        private readonly Timer timer;      // Таймер анимации

        private Bitmap buffer;

        private float frame_w;             // Ширина кадра
        private float frame_h;             // Высота кадра
        private float pix_size;            // размер "пикселя"

        private int file_index;            // индекс файла в массиве выбранных файлов
        private string file_name;          // имя загруженного файла
        private FileType file_type;        // тип загруженного файла
        private FileStream file;           // файловый поток с данными

        private int frames_num;            // количество кадров в загруженном файле
        private byte pic_width;            // реальная ширина загруженной картинки типа *.p, считанная из заголовка файла
        private byte pic_height;           // реальная высота загруженной картинки типа *.p, считанная из заголовка файла

        private int frame_s;               // размер буфера кадра
        private byte[] frame_buffer;       // буфер на кадр

        private int cut_frame_s;           // размер буфера кадра
        private byte[] cut_frame_buffer;   // буфер на кадр

        private byte[] clipboard_frame_buffer; // буфер на кадр clipboard

        private bool changing;             // флаг изменений

        private byte MATRIX_TYPE;          // тип матрицы: 0 - зигзаг, 1 - параллельная
        private byte CONNECTION_ANGLE;     // угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
        private byte STRIP_DIRECTION;      // направление ленты из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
        private byte COLOR_ORDER;          // тип порядка цветов в триаде цвета
        private byte WIDTH;                // Ширина матрицы
        private byte HEIGHT;               // высота матрицы

        private byte CUT_MATRIX_TYPE;      // тип матрицы: 0 - зигзаг, 1 - параллельная
        private byte CUT_CONNECTION_ANGLE; // угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
        private byte CUT_STRIP_DIRECTION;  // направление ленты из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
        private byte CUT_COLOR_ORDER;      // тип порядка цветов в триаде цвета
        private byte CUT_WIDTH;            // Ширина матрицы
        private byte CUT_HEIGHT;           // высота матрицы
        private byte CUT_OFFSET_X;         // Отступ обрезки по X
        private byte CUT_OFFSET_Y;         // Отступ обрезки по Y

        private bool playing;              // Флаг: вкл/выкл анимацию
        private bool save_in_porcess;      // Флаг: вкл/выкл анимацию
        private bool frame_size_ok;        // считанный из файла кадр по размеру соответствкет заданному в настройках
        private byte frame_marker;         // значения маркера разделителя кадров a читаемом файле
        private string lastError;          // Последняя ошибка - текст ошибки

        private string src_folder;         // Папка с исходными файлами
        private string dst_folder;         // Папка для сохранения обработанных файлов

        private string currentLanguage;    // Текущий язык интерфейса
        private bool inEditMode = false;   // В режиме редактирования кадра
        private bool isFrameChanged = false; // В кадр были внесены изменения

        private Cursor penCursor;
        private byte selectedColorIdx = 0;

        private ScreenColorPicker activePicker;

        public frmMain()
        {
            opt_file = Path.ChangeExtension(Application.ExecutablePath, ".opt");
            lng_file = Path.ChangeExtension(Application.ExecutablePath, ".lng");
            timer = new Timer(callback_Animate, null, Timeout.Infinite, Timeout.Infinite);

            if (File.Exists(lng_file))
            {
                try
                {
                    using (BinaryReader reader = new BinaryReader(File.Open(lng_file, FileMode.Open)))
                    {
                        currentLanguage = reader.ReadString();
                    }
                }
                catch { }
            }
            else
            {
                currentLanguage = "ru";
            }

            Thread.CurrentThread.CurrentUICulture = CultureInfo.GetCultureInfo(currentLanguage);
            Thread.CurrentThread.CurrentCulture = CultureInfo.GetCultureInfo(currentLanguage);

            InitializeComponent();

            cbMatrixType.Items.Add(Strings.mt01); // Лево-Верх строки вправо
            cbMatrixType.Items.Add(Strings.mt02); // Право-Верх строки влево
            cbMatrixType.Items.Add(Strings.mt03); // Лево-Низ строки вправо
            cbMatrixType.Items.Add(Strings.mt04); // Право-Низ строки влево
            cbMatrixType.Items.Add(Strings.mt05); // Лево-Верх зигзаг вправо
            cbMatrixType.Items.Add(Strings.mt06); // Право-Верх зигзаг влево
            cbMatrixType.Items.Add(Strings.mt07); // Лево-Низ зигзаг вправо
            cbMatrixType.Items.Add(Strings.mt08); // Право-Низ зигзаг влево
            cbMatrixType.Items.Add(Strings.mt09); // Лево-Верх колонки вниз
            cbMatrixType.Items.Add(Strings.mt10); // Право-Верх колонки вниз
            cbMatrixType.Items.Add(Strings.mt11); // Лево-Низ колонки вверх
            cbMatrixType.Items.Add(Strings.mt12); // Право-Низ колонки вверх
            cbMatrixType.Items.Add(Strings.mt13); // Лево-Верх зигзаг вниз
            cbMatrixType.Items.Add(Strings.mt14); // Право-Верх зигзаг вниз
            cbMatrixType.Items.Add(Strings.mt15); // Лево-Низ зигзаг вверх
            cbMatrixType.Items.Add(Strings.mt16); // Право-Низ зигзаг вверх

            cbCutMatrixType.Items.Add(Strings.mt01); // Лево-Верх строки вправо
            cbCutMatrixType.Items.Add(Strings.mt02); // Право-Верх строки влево
            cbCutMatrixType.Items.Add(Strings.mt03); // Лево-Низ строки вправо
            cbCutMatrixType.Items.Add(Strings.mt04); // Право-Низ строки влево
            cbCutMatrixType.Items.Add(Strings.mt05); // Лево-Верх зигзаг вправо
            cbCutMatrixType.Items.Add(Strings.mt06); // Право-Верх зигзаг влево
            cbCutMatrixType.Items.Add(Strings.mt07); // Лево-Низ зигзаг вправо
            cbCutMatrixType.Items.Add(Strings.mt08); // Право-Низ зигзаг влево
            cbCutMatrixType.Items.Add(Strings.mt09); // Лево-Верх колонки вниз
            cbCutMatrixType.Items.Add(Strings.mt10); // Право-Верх колонки вниз
            cbCutMatrixType.Items.Add(Strings.mt11); // Лево-Низ колонки вверх
            cbCutMatrixType.Items.Add(Strings.mt12); // Право-Низ колонки вверх
            cbCutMatrixType.Items.Add(Strings.mt13); // Лево-Верх зигзаг вниз
            cbCutMatrixType.Items.Add(Strings.mt14); // Право-Верх зигзаг вниз
            cbCutMatrixType.Items.Add(Strings.mt15); // Лево-Низ зигзаг вверх
            cbCutMatrixType.Items.Add(Strings.mt16); // Право-Низ зигзаг вверх

            cbMatrixType.SelectedIndex = 0;
            cbColorOrder.SelectedIndex = 0;

            activePicker = screenColorPickerFore;
            SetActivePicker(screenColorPickerFore);

            ((Bitmap)btnFill.Image) .MakeTransparent(Color.Magenta);
            ((Bitmap)btnCopy.Image) .MakeTransparent(Color.Magenta);
            ((Bitmap)btnPaste.Image).MakeTransparent(Color.Magenta);            

            lblLanguageWarning.Visible = false;
        }

        // *****************************************************************************************
        // События
        // *****************************************************************************************

        private void frmMain_Load(object sender, EventArgs e)
        {
            changing = true;
            if (File.Exists(opt_file))
            {
                try
                {
                    using (BinaryReader reader = new BinaryReader(File.Open(opt_file, FileMode.Open)))
                    {
                        edtWidth.Value = reader.ReadByte();
                        edtHeight.Value = reader.ReadByte();
                        cbMatrixType.SelectedIndex = reader.ReadSByte();
                        cbColorOrder.SelectedIndex = reader.ReadSByte();
                        trkSpeed.Value = reader.ReadInt16();
                        chkGrid.Checked = reader.ReadBoolean();

                        edtCutWidth.Value = edtWidth.Value;
                        edtCutHeight.Value = edtHeight.Value;
                        cbCutMatrixType.SelectedIndex = cbMatrixType.SelectedIndex;
                        cbCutColorOrder.SelectedIndex = cbColorOrder.SelectedIndex;
                        edtCutOffsetX.Minimum = 0;
                        edtCutOffsetX.Maximum = 127;
                        edtCutOffsetX.Value = 0;
                        edtCutOffsetY.Minimum = 0;
                        edtCutOffsetY.Maximum = 127;
                        edtCutOffsetY.Value = 0;

                        chkCutFrame.Checked = reader.ReadBoolean();
                        edtCutWidth.Value = reader.ReadByte();
                        edtCutHeight.Value = reader.ReadByte();
                        cbCutMatrixType.SelectedIndex = reader.ReadSByte();
                        cbCutColorOrder.SelectedIndex = reader.ReadSByte();
                        edtCutOffsetX.Value = reader.ReadByte();
                        edtCutOffsetY.Value = reader.ReadByte();

                        screenColorPickerFore.Color = screenColorPickerFore.BackColor = Color.FromArgb(reader.ReadInt32());
                        screenColorPickerBack.Color = screenColorPickerBack.BackColor = Color.FromArgb(reader.ReadInt32());

                        for (int i = 0; i < colorPalette.CustomColors.Count; i++)
                        {
                            colorPalette.CustomColors[i] = Color.FromArgb(reader.ReadInt32());
                        }

                        src_folder = reader.ReadString();
                        dst_folder = reader.ReadString();
                    }
                }
                catch { }
            }

            changing = true;
            cbLanguage.DataSource = new[]
            {
                CultureInfo.GetCultureInfo("ru"),
                CultureInfo.GetCultureInfo("en")
            };
            cbLanguage.DisplayMember = "NativeName";
            cbLanguage.ValueMember = "Name";

            if (!String.IsNullOrEmpty(currentLanguage))
            {
                cbLanguage.SelectedValue = currentLanguage;
            }

            colorEditor.Color = activePicker.Color;
            colorWheel.Color = activePicker.Color;

            changing = false;

            lblVersion.Text = Strings.Version + " " + AssemblyVersionInfo.CurrentVersion; // "Версия"

            calcFrameArea();

            // Загрузить список файлов из папки
            if (string.IsNullOrWhiteSpace(src_folder)) src_folder = Path.GetDirectoryName(Application.ExecutablePath);
            ScanFiles(src_folder);

            listFiles.SelectedIndex = listFiles.Items.Count > 0 ? 0 : -1;
            if (listFiles.SelectedIndex >= 0)
            {
                var item = (FileListBoxItem)listFiles.Items[listFiles.SelectedIndex];                
                openDataFile(Path.Combine(src_folder, item.FullName));
            }

            edtFolderName.Text = src_folder;

            penCursor = new Cursor(new MemoryStream(Resources.pen));

            CheckCutDimensions();
            enableControls();

            pictureBox.Refresh();
        }

        private void frmMain_FormClosing(object sender, FormClosingEventArgs e)
        {
            timer.Change(Timeout.Infinite, Timeout.Infinite);
            if (file != null) file.Close();

            // создаем объект BinaryWriter
            try
            {
                using (var writer = new BinaryWriter(File.Open(opt_file, FileMode.Create)))
                {
                    // записываем в файл значение каждого поля структуры
                    writer.Write(Convert.ToByte(edtWidth.Value));
                    writer.Write(Convert.ToByte(edtHeight.Value));
                    writer.Write(Convert.ToSByte(cbMatrixType.SelectedIndex));
                    writer.Write(Convert.ToSByte(cbColorOrder.SelectedIndex));
                    writer.Write(Convert.ToInt16(trkSpeed.Value));
                    writer.Write(chkGrid.Checked);

                    writer.Write(chkCutFrame.Checked);
                    writer.Write(Convert.ToByte(edtCutWidth.Value));
                    writer.Write(Convert.ToByte(edtCutHeight.Value));
                    writer.Write(Convert.ToSByte(cbCutMatrixType.SelectedIndex));
                    writer.Write(Convert.ToSByte(cbCutColorOrder.SelectedIndex));
                    writer.Write(Convert.ToByte(edtCutOffsetX.Value));
                    writer.Write(Convert.ToByte(edtCutOffsetY.Value));

                    writer.Write(Convert.ToInt32(screenColorPickerFore.BackColor.ToArgb()));
                    writer.Write(Convert.ToInt32(screenColorPickerBack.BackColor.ToArgb()));

                    for (int i = 0; i < colorPalette.CustomColors.Count; i++)
                    {
                        writer.Write(Convert.ToInt32(colorPalette.CustomColors[i].ToArgb()));
                    }

                    writer.Write(src_folder ?? "");
                    writer.Write(dst_folder ?? "");
                }
                using (var writer = new BinaryWriter(File.Open(lng_file, FileMode.Create)))
                {
                    writer.Write(currentLanguage);
                }
            }
            catch { }
        }

        private void frmMain_FormClosed(object sender, FormClosedEventArgs e)
        {
            if (penCursor != null)
            {
                penCursor.Dispose();
            }
        }

        private void frmMain_Resize(object sender, EventArgs e)
        {
            pictureBox.Refresh();
        }

        // Отрисовка кадра на панели
        private void panelCanvas_Paint(object sender, PaintEventArgs e)
        {
            paintFrame();
        }

        private void Dims_ValueChanged(object sender, EventArgs e)
        {
            if (changing) return;

            calcFrameArea();
            frame_buffer = LoadFrameFromFile(trkFrames.Value);
            lblWarning.Visible = !frame_size_ok && listFiles.Items.Count > 0;
            lblCurrentFrame.Text = (trkFrames.Value + 1).ToString();
            CheckCutDimensions();
            enableControls();
        }

        private void chkGrid_CheckedChanged(object sender, EventArgs e)
        {
            pictureBox.Refresh();
        }

        private void btnOpenFile_Click(object sender, EventArgs e)
        {
            var folder = "";
            using (new CenterWinDialog(this))
            {
                folder = SelectDataFolder(Path.GetDirectoryName(file_name));
            }

            if (folder == "") return;

            resetFile();
            resetFile2();

            src_folder = folder;
            edtFolderName.Text = src_folder;
            ScanFiles(folder);

            listFiles.SelectedIndex = listFiles.Items.Count > 0 ? 0 : -1;

            if (listFiles.SelectedIndex >= 0)
            {
                var item = (FileListBoxItem)listFiles.Items[listFiles.SelectedIndex];
                openDataFile2(Path.Combine(src_folder, item.FullName));
                frame_buffer = LoadFrameFromFile(trkFrames.Value);
                lblWarning.Visible = !frame_size_ok && listFiles.Items.Count > 0;
            }

            lblCurrentFrame.Text = (trkFrames.Value + 1).ToString();
            edtFolderName.Text = src_folder;

            changing = false;

            enableControls();
            pictureBox.Refresh();

            SetTitle();
        }

        private void cbMatrixType_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (changing) return;
            // Настройка параметров матрицы для пересчета позиции x,y в номер элемента (триплета)
            decodeMatrixType(cbMatrixType.SelectedIndex);
            pictureBox.Refresh();
        }

        private void cbColorOrder_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (changing) return;
            COLOR_ORDER = (byte)cbColorOrder.SelectedIndex;
            pictureBox.Refresh();
        }

        private void cbCutMatrixType_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (changing) return;
            decodeCutMatrixType(cbCutMatrixType.SelectedIndex);
            CheckCutDimensions();
        }

        private void cbCutColorOrder_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (changing) return;
            CUT_COLOR_ORDER = (byte)cbCutColorOrder.SelectedIndex;
            CheckCutDimensions();
        }

        private void trkFrames_ValueChanged(object sender, EventArgs e)
        {
            if (changing) return;
            frame_buffer = LoadFrameFromFile(trkFrames.Value);
            lblCurrentFrame.Text = (trkFrames.Value + 1).ToString();
            lblWarning.Visible = !frame_size_ok && listFiles.Items.Count > 0;
            paintFrame();
        }

        private void pnlCanvas_Resize(object sender, EventArgs e)
        {
            calcFrameSize();
            pictureBox.Refresh();
        }

        private void trkSpeed_ValueChanged(object sender, EventArgs e)
        {
            if (playing)
            {
                timer.Change(1, trkSpeed.Maximum - trkSpeed.Value + trkSpeed.Minimum);
            }
        }

        private void callback_Animate(object state)
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker) (() => callback_Animate(state))); }
                catch { }

                return;
            }

            var idx = trkFrames.Value + 1;
            if (idx >= edtFragEnd.Value)
            {
                idx = Convert.ToInt32(edtFragStart.Value) - 1;

                // Загрузить следующий файл из списка выбранных
                if (listFiles.SelectedItems.Count > 1)
                {
                    file_index++;
                    if (file_index >= listFiles.SelectedItems.Count)
                    {
                        file_index = 0;
                    }

                    var item = (FileListBoxItem) listFiles.SelectedItems[file_index];
                    openDataFile(Path.Combine(src_folder, item.FullName));
                    SetTitle();
                }
            }

            trkFrames.Value = idx;

            pictureBox.Refresh();            
        }

        private void btnStart_Click(object sender, EventArgs e)
        {
            file_index = 0;
            playing = true;
            disableControls();
            
            timer.Change(1, trkSpeed.Maximum - trkSpeed.Value + trkSpeed.Minimum);            
            flowLayoutPanel1.Invalidate();
        }

        private void btnStop_Click(object sender, EventArgs e)
        {
            playing = false;
            enableControls();
            timer.Change(Timeout.Infinite, Timeout.Infinite);

            if (listFiles.SelectedItems.Count > 1 && listFiles.SelectedIndex >= 0)
            {
                var item = (FileListBoxItem)listFiles.Items[listFiles.SelectedIndex];
                resetFile();
                openDataFile(Path.Combine(src_folder, item.FullName));
            }

            flowLayoutPanel1.Invalidate();
        }

        private void btnFragStart_Click(object sender, EventArgs e)
        {
            edtFragStart.Value = trkFrames.Value + 1;
            if (edtFragEnd.Value < edtFragStart.Value)
            {
                edtFragEnd.Value = edtFragStart.Value;
            }
        }

        private void btnFragEnd_Click(object sender, EventArgs e)
        {
            edtFragEnd.Value = trkFrames.Value + 1;
            if (edtFragStart.Value > edtFragEnd.Value)
            {
                edtFragStart.Value = edtFragEnd.Value;
            }
        }

        private void edtFragStart_ValueChanged(object sender, EventArgs e)
        {
            if (edtFragEnd.Value < edtFragStart.Value)
            {
                edtFragEnd.Value = edtFragStart.Value;
            }
        }

        private void edtFragEnd_ValueChanged(object sender, EventArgs e)
        {
            if (edtFragStart.Value > edtFragEnd.Value)
            {
                edtFragStart.Value = edtFragEnd.Value;
            }
        }

        private void btnFragSave_Click(object sender, EventArgs e)
        {
            saveFileDialog.Title = Strings.SaveJinxMovieFile; // "Сохранить фрагмент в файл"
            saveFileDialog.FileName = file_name;
            saveFileDialog.Filter = Strings.JinxMovieFilter;  // "Jinx Movie|*.out";
            saveFileDialog.FilterIndex = 0;

            var res = saveFileDialog.ShowDialog();
            if (res == DialogResult.OK)
            {
                var new_file_name = saveFileDialog.FileName;
                var same = new_file_name.ToLower() == file_name.ToLower();
                var tmp_file_name = new_file_name + ".new";

                try
                {
                    // Сохранить фрагмент в новый временный файл
                    using (BinaryWriter writer = new BinaryWriter(File.Open(tmp_file_name, FileMode.Create)))
                    {
                        var start_frame = Convert.ToInt32(edtFragStart.Value) - 1;
                        var end_frame = Convert.ToInt32(edtFragEnd.Value) - 1;
                        var num_frames = end_frame - start_frame + 1;

                        long position = start_frame * (frame_s + 1);
                        long length = num_frames * (frame_s + 1);
                        file.Position = position;

                        for (long i = 0; i < length; i++)
                        {
                            var b = Convert.ToByte(file.ReadByte());
                            writer.Write(b);
                        }
                    }

                    // Если файл тот же самый - удалить старый файл
                    if (same)
                    {
                        file.Close();
                        file = null;
                        File.Delete(file_name);
                    }

                    // Переименовать временный файл в постоянное имя
                    if (File.Exists(new_file_name))
                    {
                        File.Delete(new_file_name);
                    }

                    File.Move(tmp_file_name, new_file_name);

                    // Добавить сохраненный файл в список файлов (пересканировать папку)
                    var save_idx = listFiles.SelectedIndex;
                    changing = true;
                    ScanFiles(src_folder);
                    changing = false;
                    if (save_idx < 0 || save_idx >= listFiles.Items.Count) save_idx = 0;
                    listFiles.SelectedIndex = save_idx;
                }
                catch (Exception ex)
                {
                    using (new CenterWinDialog(this))
                    {
                        MessageBox.Show(Strings.SaveFileError + " '" + new_file_name + "'\n\n" + ex.Message, Strings.Error, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }
            }
        }

        private void btnFrameSave_Click(object sender, EventArgs e)
        {
            saveFileDialog.Title = Strings.SaveFrameToFile; // "Сохранить кадр в файл"
            saveFileDialog.FileName = Path.ChangeExtension(file_name, ".p");
            saveFileDialog.Filter = Strings.FrameFileFilter;  // "Picture file|*.p";
            saveFileDialog.FilterIndex = 0;

            var res = saveFileDialog.ShowDialog();
            if (res == DialogResult.OK)
            {
                var new_file_name = saveFileDialog.FileName;

                if (File.Exists(new_file_name))
                {
                    using (new CenterWinDialog(this))
                    {
                        res = MessageBox.Show(Strings.FileExists + " '" + new_file_name + "'\n\n" + Strings.Overwrite, Strings.Error, MessageBoxButtons.OKCancel, MessageBoxIcon.Question);
                        if (res != DialogResult.OK) return;
                    }
                }

                // Сохранить текущий кадр в новый файл
                byte b1, b2, b3;
                int idx;
                Color color;

                var buff = new byte[frame_s];
                Array.Clear(buff, 0, frame_s);

                for (byte x = 0; x < WIDTH; x++)
                {
                    for (byte y = 0; y < HEIGHT; y++)
                    {
                        idx = getPixelNumber(x, y) * 3;
                        b1 = frame_buffer[idx];
                        b2 = frame_buffer[idx + 1];
                        b3 = frame_buffer[idx + 2];
                        color = getColor(b1, b2, b3, COLOR_ORDER);
                        color = Color.FromArgb(ColorConverter.gammaCorrection(color.ToArgb()));
                        
                        idx = (x * HEIGHT + y) * 3;
                        buff[idx] = color.R;
                        buff[idx + 1] = color.G;
                        buff[idx + 2] = color.B;
                    }
                }

                FileStream out_file = null;
                try
                {
                    out_file = File.Open(new_file_name, FileMode.Create);
                    using (BinaryWriter writer = new BinaryWriter(out_file))
                    {
                        // Заголовок файла картинки
                        writer.Write((byte) 0x33);
                        writer.Write(WIDTH);
                        writer.Write(HEIGHT);
                        for (int i = 0; i < frame_s; i++)
                        {
                            writer.Write(buff[i]);
                        }
                    }

                    // Добавить сохраненный файл в список файлов (пересканировать папку)
                    var save_idx = listFiles.SelectedIndex;
                    changing = true;
                    ScanFiles(src_folder);
                    changing = false;
                    if (save_idx < 0 || save_idx >= listFiles.Items.Count) save_idx = 0;
                    listFiles.SelectedIndex = save_idx;
                }
                catch (Exception ex)
                {
                    using (new CenterWinDialog(this))
                    {
                        MessageBox.Show(Strings.SaveFileError + " '" + new_file_name + "'\n\n" + ex.Message, Strings.Error, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }
                finally
                {
                    if (out_file != null) out_file.Close();
                }
            }
        }

        private void btnFragCodeSave_Click(object sender, EventArgs e)
        {
            saveFileDialog.Title = Strings.SaveFragmentToCode; // "Сохранить фрагмент в код"
            saveFileDialog.FileName = Path.ChangeExtension(file_name, ".h");
            saveFileDialog.Filter = Strings.FragmentCodeFilter;  // "Файл с кодом анимации|*.h";
            saveFileDialog.FilterIndex = 0;

            var res = saveFileDialog.ShowDialog();
            if (res == DialogResult.OK)
            {
                var new_file_name = saveFileDialog.FileName;

                if (File.Exists(new_file_name))
                {
                    using (new CenterWinDialog(this))
                    {
                        res = MessageBox.Show(Strings.FileExists + " '" + new_file_name + "'\n\n" + Strings.Overwrite, Strings.Error, MessageBoxButtons.OKCancel, MessageBoxIcon.Question);
                        if (res != DialogResult.OK) return;
                    }
                }

                var start_frame = Convert.ToInt32(edtFragStart.Value) - 1;
                var end_frame = Convert.ToInt32(edtFragEnd.Value) - 1;

                var ok = SaveFragment(new_file_name, start_frame, end_frame);
                if (!ok)
                {
                    using (new CenterWinDialog(this))
                    {
                        MessageBox.Show(Strings.SaveFileError + " '" + file_name + "'\n\n" + lastError, Strings.Error, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }
            }
        }

        private void btnFrameCodeSave_Click(object sender, EventArgs e)
        {
            saveFileDialog.Title = Strings.SaveFrameToCode; // "Сохранить кадр в код"
            saveFileDialog.FileName = Path.ChangeExtension(file_name, ".h");
            saveFileDialog.Filter = Strings.FrameCodeFilter;  // "Файл с кодом картинки|*.h";
            saveFileDialog.FilterIndex = 0;

            var res = saveFileDialog.ShowDialog();
            if (res == DialogResult.OK)
            {
                var new_file_name = saveFileDialog.FileName;

                if (File.Exists(new_file_name))
                {
                    using (new CenterWinDialog(this))
                    {
                        res = MessageBox.Show(Strings.FileExists + " '" + new_file_name + "'\n\n" + Strings.Overwrite, Strings.Error, MessageBoxButtons.OKCancel, MessageBoxIcon.Question);
                        if (res != DialogResult.OK) return;
                    }
                }
                
                var ok = SaveFragment(new_file_name, trkFrames.Value, trkFrames.Value, false);
                if (!ok)
                {
                    using (new CenterWinDialog(this))
                    {
                        MessageBox.Show(Strings.SaveFileError + " '" + file_name + "'\n\n" + lastError, Strings.Error, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }
            }
        }

        private bool SaveFragment(string file_name, int start_frame, int end_frame, bool createAnimationStruct = true)
        {
            var ok = true;
            StreamWriter sw = null;

            var var_name = Path.GetFileNameWithoutExtension(file_name) ?? "pic";
            const string wrong = " ?!\"#$%^&*()-`~.,:'></\\{}|";

            if (var_name.IndexOfAny(wrong.ToCharArray()) >= 0)
            {
                foreach (var ch in wrong) var_name = var_name.Replace(ch, '_');
            }

            var ani_name = var_name;
            if ("0123456789".IndexOf(var_name[0]) >= 0)
            {
                var_name = "fr_" + var_name;
            }

            try
            {
                var num_frames = end_frame - start_frame + 1;
                var code_len = num_frames.ToString().Length;

                // Сохранить текущий кадр в файл кода
                var line = new StringBuilder();
                var sb = new StringBuilder();
                var enc = new UTF8Encoding(false);

                sw = new StreamWriter(file_name, false, enc) { AutoFlush = true };
                sb.AppendLine(Strings.FrameCodeHeader);
                sw.WriteLine(sb.ToString());

                var cnt = 0;

                for (int fr = start_frame; fr <= end_frame; fr++)
                {
                    var buff = LoadFrameFromFile(fr);
                    sb.Clear();
                    sb.AppendFormat("const uint16_t {0}_{1}[] PROGMEM = {{\n", var_name, cnt.ToString().PadLeft(code_len, '0'));
                    cnt++;

                    for (int y = HEIGHT - 1; y >= 0; y--)
                    {
                        line.Clear();
                        line.Append("    ");

                        for (int x = 0; x < WIDTH; x++)
                        {
                            int idx;
                            byte b1, b2, b3;
                            Color color;
                            if (file_type == FileType.MOVIE)
                            {
                                idx = getPixelNumber((byte) x, (byte) y) * 3;
                                b1 = buff[idx];
                                b2 = buff[idx + 1];
                                b3 = buff[idx + 2];
                                color = getColor(b1, b2, b3, COLOR_ORDER);
                            }
                            else
                            {
                                idx = (x * HEIGHT + y) * 3;
                                b1 = buff[idx];
                                b2 = buff[idx + 1];
                                b3 = buff[idx + 2];
                                color = Color.FromArgb(b1, b2, b3);
                            }
                            line.AppendFormat("0x{0:x4}, ", ColorConverter._888to565(color));
                        }

                        var s = line.ToString();
                        if (y == 0) s = s.Substring(0, s.Length - 2);
                        sb.AppendLine(s);
                    }
                    sb.AppendLine("};");
                    sw.WriteLine(sb.ToString());
                }

                var sbf = new StringBuilder("{ ");
                cnt = 0;                
                for (int fr = 0; fr < num_frames; fr++)
                {
                    var last_frame = cnt == num_frames - 1;
                    var frame_name = string.Format("{0}_{1}{2}", var_name, cnt.ToString().PadLeft(code_len, '0'), (last_frame ? "" : ", "));
                    sbf.Append(frame_name);
                    sb .Append(frame_name);                    
                    cnt++;
                }
                sbf.Append(" }");
                sb.Clear();
                sb.AppendFormat("const uint16_t* const {0}_array[{1}] PROGMEM = {2};\n", var_name, num_frames, sbf);
                sw.WriteLine(sb.ToString());

                if (createAnimationStruct)
                {
                    sb.Clear();
                    sb.AppendFormat("const animation_t animation_{0} PROGMEM = {{\n", ani_name);
                    sb.AppendLine("   .frames  = " + sbf + ",");
                    sb.AppendLine("   .start_x = 0,                        // Позиция отображения X (начальная)");
                    sb.AppendLine("   .start_y = 0,                        // Позиция отображения Y (начальная)");
                    sb.AppendLine("   .options = 1+2+4+8+16,               // Битовый флаг дополнительных параметров картинки");
                    sb.AppendLine("                                        //   1 - центрировать по горизонтали  (позиция start_x игнорируется)");
                    sb.AppendLine("                                        //   2 - центрировать по вертикали  (позиция start_y игнорируется)");
                    sb.AppendLine("                                        //   4 - есть прозрачные пиксели, цвет прозрачности в transparent_color");
                    sb.AppendLine("                                        //   8 - перед отрисовкой ПЕРВОГО кадра закрашивать всю матрицу цветом background_first_color");
                    sb.AppendLine("                                        //  16 - перед отрисовкой СЛЕДУЮЩИХ кадров закрашивать всю матрицу цветом background_color");
                    sb.AppendLine("                                        //  64 - начальное отображение - зеркальное по оси х");
                    sb.AppendLine("                                        // 128 - начальное отображение - зеркальное по оси y");
                    sb.AppendFormat("   .frame_width = {0},                   // Ширина картинки (фрейма)\n", WIDTH);
                    sb.AppendFormat("   .frame_height = {0},                  // Высота картинки (фрейма)\n", HEIGHT);
                    sb.AppendLine("   .row_draw_direction = 0,             // Направление построчного рисования (битовые флаги):");
                    sb.AppendLine("                                        // Биты 0 и 1 (маска 0x03)");
                    sb.AppendLine("                                        //   0 (0x00) - сверху вниз");
                    sb.AppendLine("                                        //   1 (0x01) - снизу вверх");
                    sb.AppendLine("                                        //   2 (0x10) - слева направо");
                    sb.AppendLine("                                        //   3 (0x11)- справа налево");
                    sb.AppendLine("   .draw_frame_interval = 125,          // Интервал отрисовки очередной порции картинки анимации (строка при построчной анимации или кадр при покадровой)");
                    sb.AppendLine("   .draw_row_interval = 0,              // Задержка мс между отрисовкой строк изображения (если 0 - рисуется кадр целиком, а не построчно)");
                    sb.AppendLine("   .move_x_interval = 0,                // Смещение по оси X каждые N мс");
                    sb.AppendLine("   .move_y_interval = 0,                // Смещение по оси Y каждые N мс");
                    sb.AppendLine("   .move_type = 0,                      // Тип движения - битовый флаг (разрешено только при отрисовке кадра целиком, при построчном - не работает):");
                    sb.AppendLine("                                        //   0 - нет;");
                    sb.AppendLine("                                        //   1 - из начальной позиции вправо");
                    sb.AppendLine("                                        //   2 - из начальной позиции влево");
                    sb.AppendLine("                                        //   4 - из начальной позиции вверх");
                    sb.AppendLine("                                        //   8 - из начальной позиции вниз");
                    sb.AppendLine("                                        //  16 - отражение при достижении границы по горизонтали");
                    sb.AppendLine("                                        //  32 - отражение при достижении границы по вертикали");
                    sb.AppendLine("                                        //  64 - переворот картинки при отражении по горизонтали");
                    sb.AppendLine("                                        // 128 - переворот картинки при отражении по вертикали");
                    sb.AppendLine("                                        // 256 - 1 - выходить за боковые рамки; 0 - не выходить");
                    sb.AppendLine("                                        // 512 - 1 - выходить за верх/низ; 0 - не выходить");
                    sb.AppendLine("   .transparent_color = 0x000000,       // Этот цвет - прозрачный, пиксели этого цвета не рисуются");
                    sb.AppendLine("   .background_first_color = 0x000000,  // Цвет заливки ВСЕЙ матрицы перед тем, как рисовать самый первый фрейм при активации эффекта анимации");
                    sb.AppendLine("   .background_color = 0x000000         // Цвет заливки ВСЕЙ матрицы перед тем, как рисовать очередной фрейм");
                    sb.AppendLine("};");
                    sb.AppendLine();
                    sw.WriteLine(sb.ToString());
                }
            }
            catch (Exception ex)
            {
                ok = false;
                lastError = ex.Message;
            }
            finally
            {
                if (sw != null)
                {
                    sw.Flush();
                    sw.Close();
                }
            }
            return ok;
        }

        private void listFiles_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (changing || listFiles.Items.Count == 0) return;

            resetFile();
            var idx = listFiles.SelectedIndex;
            if (idx < 0) idx = 0;

            var item = (FileListBoxItem)listFiles.Items[idx];
            openDataFile(Path.Combine(src_folder, item.FullName));

            // Если это файл картинки (*.p) и его размеры неизвестны - считать из файла размер картинки
            if (item.Type == FileType.PICTURE)
            {
                if ((item.Width == 0 || item.Height == 0) && pic_width != 0 && pic_height != 0)
                {
                    item.Width = pic_width;
                    item.Height = pic_height;
                }
                changing = true;
                edtWidth.Value = pic_width;
                edtHeight.Value = pic_height;
                changing = false;
                if (WIDTH != pic_width || HEIGHT != pic_height)
                {
                    calcFrameArea();
                }
            }

            Text = String.Format("Jinx Frame Viewer [{0}]", item.FullName);
            
            enableControls();
        }

        private void listFiles_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            btnStart_Click(null, null);
        }

        private void chkCutFrame_CheckedChanged(object sender, EventArgs e)
        {
            pnlCutFrame.Enabled = chkCutFrame.Checked;
            btnStartCut.Enabled = chkCutFrame.Checked;
        }

        private void edtCutWidth_ValueChanged(object sender, EventArgs e)
        {
            CheckCutDimensions();
        }

        private void edtCutHeight_ValueChanged(object sender, EventArgs e)
        {
            CheckCutDimensions();
        }

        private void edtCutOffsetX_ValueChanged(object sender, EventArgs e)
        {
            CheckCutDimensions();
        }

        private void edtCutOffsetY_ValueChanged(object sender, EventArgs e)
        {
            CheckCutDimensions();
        }

        private void pnlCanvas_Resize_1(object sender, EventArgs e)
        {
            pictureBox.Width = pnlCanvas.Width;
            pictureBox.Height = pnlCanvas.Height;
            pictureBox.MinimumSize = new Size(pnlCanvas.Width, pnlCanvas.Height);
        }

        private void btnStartCut_Click(object sender, EventArgs e)
        {
            if (save_in_porcess)
            {
                btnStartCut.Text = Strings.Stopping; // "Остановка. Ждите...";
                btnStartCut.Enabled = false;
                save_in_porcess = false;
                return;
            }

            // Запрос папки сохранения
            using (new CenterWinDialog(this))
            {
                dst_folder = SelectDataFolder(string.IsNullOrEmpty(dst_folder) ? src_folder : dst_folder, Strings.ChooseSaveFolder);
            }

            if (dst_folder == "") return;

            if (dst_folder == src_folder)
            {
                using (new CenterWinDialog(this))
                {
                    MessageBox.Show(Strings.WrongDestinationFolder, Strings.Error, MessageBoxButtons.OK, MessageBoxIcon.Error);
                }

                return;
            }

            decodeMatrixType(cbMatrixType.SelectedIndex);
            decodeCutMatrixType(cbCutMatrixType.SelectedIndex);

            // Выделение памяти под буфер кадра обрезки
            cut_frame_s = CUT_WIDTH * CUT_HEIGHT * 3;
            cut_frame_buffer = new byte[cut_frame_s];

            save_in_porcess = true;

            btnStartCut.Text = Strings.StopAction; // "Остановить";

            // Список файлов для обработки
            var sl = new List<string>();
            foreach (var item in listFiles.SelectedItems)
            {
                if (((FileListBoxItem) item).Type == FileType.MOVIE)
                {
                    sl.Add(item.ToString());
                }
            }

            if (sl.Count == 0)
            {
                using (new CenterWinDialog(this))
                {
                    MessageBox.Show(Strings.NoMovieFiles, Strings.Error, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }
            }

            disableControls();

            btnStartCut.Enabled = true;
            btnStop.Enabled = false;
            trkFrames.Enabled = false;
            trkSpeed.Enabled = false;

            // Отдельный поток в котором будет выполняться обработка файлов для перекодирования
            var thread = new Thread(o => processCutFiles(src_folder, dst_folder, sl));
            thread.SetApartmentState(ApartmentState.MTA);
            thread.IsBackground = true;

            thread.Start();
        }
        
        // *****************************************************************************************
        // Методы
        // *****************************************************************************************

        private void resetFile()
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => resetFile())); } catch { }
                return;
            }

            playing = false;
            lblWarning.Visible = false;
            btnStart.Enabled = false;
            btnStop.Enabled = false;
            lblFirstFrame.Visible = false;
            lblCurrentFrame.Visible = false;
            lblTotalFrames.Visible = false;
            trkFrames.Enabled = false;
            frames_num = 0;

            edtFragStart.Minimum = 0;
            edtFragStart.Value = 0;
            edtFragStart.Maximum = 1;
            edtFragEnd.Minimum = 0;
            edtFragEnd.Value = 0;
            edtFragEnd.Maximum = 1;

            edtFragStart.Enabled = false;
            edtFragEnd.Enabled = false;
            btnFragStart.Enabled = false;
            btnFragEnd.Enabled = false;
            btnFragSave.Enabled = false;

            changing = true;
            trkFrames.Value = 0;
            changing = false;

            timer.Change(Timeout.Infinite, Timeout.Infinite);
            if (file != null)
            {
                file.Close();
                file = null;
            }
        }

        private void resetFile2()
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => resetFile2())); } catch { }
                return;
            }

            edtFolderName.Text = "";
            changing = true;
            listFiles.Items.Clear();
            changing = false;
        }

        private void calcFrameArea()
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => calcFrameArea())); } catch { }
                return;
            }

            WIDTH = Convert.ToByte(edtWidth.Value);
            HEIGHT = Convert.ToByte(edtHeight.Value);
            COLOR_ORDER = (byte)cbColorOrder.SelectedIndex;

            frame_s = WIDTH * HEIGHT * 3;

            decodeMatrixType(cbMatrixType.SelectedIndex);
            calcFrameSize();

            if (file != null)
            {
                frames_num = file_type == FileType.MOVIE ? Convert.ToInt32(file.Length / (frame_s + 1)) : 1;

                lblTotalFrames.Text = frames_num.ToString();
                trkFrames.Maximum = frames_num - 1;
                lblCurrentFrame.Text = (trkFrames.Value + 1).ToString();

                frame_buffer = LoadFrameFromFile(trkFrames.Value);
            }
        }

        private void calcFrameSize()
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => calcFrameSize())); } catch { }
                return;
            }

            var w = (pnlCanvas.Width - 2 * margin) / (double) WIDTH;
            var h = (pnlCanvas.Height - 2 * margin) / (double) HEIGHT;

            pix_size = Convert.ToSingle(Math.Min(w, h));
            frame_w = pix_size * Convert.ToSingle(WIDTH);
            frame_h = pix_size * Convert.ToSingle(HEIGHT);

            buffer = new Bitmap(pnlCanvas.Width, pnlCanvas.Height);
        }

        private byte[] LoadFrameFromFile(int num)
        {
            int cnt;
            byte[] buff = null;
            var ok = true;
            try
            {

                if (file_type == FileType.MOVIE)
                {
                    // Загрузить ролик из файла формата Jinx "*.out"
                    // читать первый байт кадра - это маркер начала фрейма / канала
                    // В начале файла - всегда верный маркер. Если читать в середине и 
                    // размеры матрицы заданы неправильно - будет считан неправильный байт маркера -
                    // поэтому запоминаем только в самом начале файла один раз
                    long position = num * (frame_s + 1);
                    file.Position = position;

                    var b = (byte)file.ReadByte();
                    if (position == 0) frame_marker = b;

                    // Размер кадра - W*H*3 - 3 байта на цвет точки
                    frame_s = WIDTH * HEIGHT * 3;
                    buff = new byte[frame_s];
                    Array.Clear(buff, 0, frame_s);
                    cnt = file.Read(buff, 0, frame_s);

                    // Следующий байт в кадре должен иметь метку начала кадра
                    int mark2 = frame_marker;
                    if (file.Position < file.Length - 1)
                    {
                        mark2 = file.ReadByte();
                    }

                    pic_width = 0;
                    pic_height = 0;
                    frame_size_ok = cnt == frame_s && frame_marker == mark2;
                }
                else
                {
                    // Загрузить картинку из файла формата сохраненной картинки пользователя "*.p"
                    file.Position = 0;

                    // Считать размер кадра картинки
                    var sign = (byte) file.ReadByte(); // Сигнатура картинки 0x33
                    pic_width = (byte) file.ReadByte(); // Ширина картинки
                    pic_height = (byte) file.ReadByte(); // Высота картинки

                    // Размер кадра - W*H*3 - 3 байта на цвет точки
                    frame_s = pic_width * pic_height * 3;
                    buff = new byte[frame_s];
                    Array.Clear(buff, 0, frame_s);

                    if (sign == 0x33) { 
                        cnt = file.Read(buff, 0, frame_s);
                        frame_size_ok = cnt == frame_s;
                    }
                    else
                    {
                        frame_size_ok = false;
                        throw new Exception(Strings.WrongDataInFile);
                    }
                }
                
                // Гамма-коррекция цветов считанного кадра для более естественного 
                // отображения цветов светодиода на плашке редактора в программе                
                for (int i = 0; i < WIDTH * HEIGHT; i++)
                {
                    var idx = i * 3;
                    var r = buff[idx];
                    var g = buff[idx + 1];
                    var b = buff[idx + 2];

                    var color = Color.FromArgb(ColorConverter.gammaCorrectionBack(Color.FromArgb(r, g, b).ToArgb()));

                    buff[idx] = color.R;
                    buff[idx + 1] = color.G;
                    buff[idx + 2] = color.B;
                }
            }
            catch (Exception ex)
            {
                lastError = Strings.FrameLoadError + " '" + num + "'\n\n" + ex.Message; // "Ошибка загрузки кадра"
                ok = false;
            }
            return ok ? buff : new byte[0];
        }

        private bool SaveFrameToFile(int num)
        {
            var ok = true;
            try
            {
                if (file != null) file.Close();

                // Сохранить фрагмент в новый временный файл
                file = File.Open(file_name, FileMode.Open, FileAccess.ReadWrite);
                using (var writer = new BinaryWriter(file))
                {
                    file.Position = (file_type == FileType.MOVIE) 
                        ? num * (frame_s + 1) + 1 // Пропускаем маркер разделителя кадров                    
                        : file.Position = 3;      // Пропускаем заголовок файла картинки

                    if (file_type == FileType.MOVIE)
                    {
                        for (int i = 0; i < WIDTH * HEIGHT; i++)
                        {
                            int idx = i * 3;
                            var b1 = frame_buffer[idx];
                            var b2 = frame_buffer[idx + 1];
                            var b3 = frame_buffer[idx + 2];

                            var color = getColor(b1, b2, b3, COLOR_ORDER);
                            color = Color.FromArgb(ColorConverter.gammaCorrection(color.ToArgb()));

                            var bytes = getBytes(color, COLOR_ORDER);

                            writer.Write(bytes[0]);
                            writer.Write(bytes[1]);
                            writer.Write(bytes[2]);
                        }
                    }
                    else
                    {
                        // Сохранить буфер кадра
                        for (byte x = 0; x < WIDTH; x++)
                        {
                            for (byte y = 0; y < HEIGHT; y++)
                            {
                                int idx = (x * HEIGHT + y) * 3;
                                var b1 = frame_buffer[idx];
                                var b2 = frame_buffer[idx + 1];
                                var b3 = frame_buffer[idx + 2];
                                var color = Color.FromArgb(b1, b2, b3);
                                color = Color.FromArgb(ColorConverter.gammaCorrection(color.ToArgb()));
                                writer.Write(color.R);
                                writer.Write(color.G);
                                writer.Write(color.B);
                            }
                        }
                    }

                    writer.Close();
                }
            }
            catch (Exception ex)
            {
                ok = false;
                lastError = ex.Message;
            }
            finally
            {
                if (file != null)
                {
                    file.Close();
                    file = null;
                }
            }

            // Reopen file
            
            ok &= openDataFile2(file_name);
            if (ok)
            {
                frame_buffer = LoadFrameFromFile(num);
                ok = frame_buffer.Length > 0;
            }

            RefreshPictureBox();

            return ok;
        }

        private void decodeMatrixType(int type)
        {
            /*
                MATRIX_TYPE;      // тип матрицы: 0 - зигзаг, 1 - параллельная
                CONNECTION_ANGLE; // угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
                STRIP_DIRECTION;  // направление ленты из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
             */
            switch (type)
            {
                case 0: // Лево-Верх строки вправо
                    MATRIX_TYPE = 1;
                    CONNECTION_ANGLE = 1;
                    STRIP_DIRECTION = 0;
                    break;
                case 1: // Право-Верх строки влево
                    MATRIX_TYPE = 1;
                    CONNECTION_ANGLE = 2;
                    STRIP_DIRECTION = 2;
                    break;
                case 2: // Лево-Низ строки вправо
                    MATRIX_TYPE = 1;
                    CONNECTION_ANGLE = 0;
                    STRIP_DIRECTION = 0;
                    break;
                case 3: // Право-Низ строки влево
                    MATRIX_TYPE = 1;
                    CONNECTION_ANGLE = 3;
                    STRIP_DIRECTION = 2;
                    break;
                case 4: // Лево-Верх зигзаг вправо
                    MATRIX_TYPE = 0;
                    CONNECTION_ANGLE = 1;
                    STRIP_DIRECTION = 0;
                    break;
                case 5: // Право-Верх зигзаг влево
                    MATRIX_TYPE = 0;
                    CONNECTION_ANGLE = 2;
                    STRIP_DIRECTION = 2;
                    break;
                case 6: // Лево-Низ зигзаг вправо
                    MATRIX_TYPE = 0;
                    CONNECTION_ANGLE = 0;
                    STRIP_DIRECTION = 0;
                    break;
                case 7: // Право-Низ зигзаг влево
                    MATRIX_TYPE = 0;
                    CONNECTION_ANGLE = 3;
                    STRIP_DIRECTION = 2;
                    break;
                case 8: // Лево-Верх колонки вниз
                    MATRIX_TYPE = 1;
                    CONNECTION_ANGLE = 1;
                    STRIP_DIRECTION = 3;
                    break;
                case 9: // Право-Верх колонки вниз
                    MATRIX_TYPE = 1;
                    CONNECTION_ANGLE = 2;
                    STRIP_DIRECTION = 3;
                    break;
                case 10: // Лево-Низ колонки вверх
                    MATRIX_TYPE = 1;
                    CONNECTION_ANGLE = 0;
                    STRIP_DIRECTION = 1;
                    break;
                case 11: // Право-Низ колонки вверх
                    MATRIX_TYPE = 1;
                    CONNECTION_ANGLE = 3;
                    STRIP_DIRECTION = 1;
                    break;
                case 12: // Лево-Верх зигзаг вниз
                    MATRIX_TYPE = 0;
                    CONNECTION_ANGLE = 1;
                    STRIP_DIRECTION = 3;
                    break;
                case 13: // Право-Верх зигзаг вниз
                    MATRIX_TYPE = 0;
                    CONNECTION_ANGLE = 2;
                    STRIP_DIRECTION = 3;
                    break;
                case 14: // Лево-Низ зигзаг вверх
                    MATRIX_TYPE = 0;
                    CONNECTION_ANGLE = 0;
                    STRIP_DIRECTION = 1;
                    break;
                case 15: // Право-Низ зигзаг вверх
                    MATRIX_TYPE = 0;
                    CONNECTION_ANGLE = 3;
                    STRIP_DIRECTION = 1;
                    break;
            }
        }

        // получить номер пикселя в ленте по координатам
        private int getPixelNumber(byte x, byte y)
        {
            var xx = THIS_X(x, y);
            var yy = THIS_Y(x, y);
            var ww = THIS_W(x, y);

            return (yy % 2 == 0 || MATRIX_TYPE == 1)
                ? yy * ww + xx // если чётная строка
                : yy * ww + ww - xx - 1; // если нечётная строка
        }

        private byte THIS_X(byte x, byte y)
        {
            /*
                CONNECTION_ANGLE; // угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
                STRIP_DIRECTION;  // направление ленты из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
             */
            if (CONNECTION_ANGLE == 0 && STRIP_DIRECTION == 0) return x;
            if (CONNECTION_ANGLE == 0 && STRIP_DIRECTION == 1) return y;
            if (CONNECTION_ANGLE == 1 && STRIP_DIRECTION == 0) return x;
            if (CONNECTION_ANGLE == 1 && STRIP_DIRECTION == 3) return Convert.ToByte(HEIGHT - y - 1);
            if (CONNECTION_ANGLE == 2 && STRIP_DIRECTION == 2) return Convert.ToByte(WIDTH - x - 1);
            if (CONNECTION_ANGLE == 2 && STRIP_DIRECTION == 3) return Convert.ToByte(HEIGHT - y - 1);
            if (CONNECTION_ANGLE == 3 && STRIP_DIRECTION == 2) return Convert.ToByte(WIDTH - x - 1);
            if (CONNECTION_ANGLE == 3 && STRIP_DIRECTION == 1) return y;
            return x;
        }

        private byte THIS_Y(byte x, byte y)
        {
            /*
                CONNECTION_ANGLE; // угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
                STRIP_DIRECTION;  // направление ленты из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
             */
            if (CONNECTION_ANGLE == 0 && STRIP_DIRECTION == 0) return y;
            if (CONNECTION_ANGLE == 0 && STRIP_DIRECTION == 1) return x;
            if (CONNECTION_ANGLE == 1 && STRIP_DIRECTION == 0) return Convert.ToByte(HEIGHT - y - 1);
            if (CONNECTION_ANGLE == 1 && STRIP_DIRECTION == 3) return x;
            if (CONNECTION_ANGLE == 2 && STRIP_DIRECTION == 2) return Convert.ToByte(HEIGHT - y - 1);
            if (CONNECTION_ANGLE == 2 && STRIP_DIRECTION == 3) return Convert.ToByte(WIDTH - x - 1);
            if (CONNECTION_ANGLE == 3 && STRIP_DIRECTION == 2) return y;
            if (CONNECTION_ANGLE == 3 && STRIP_DIRECTION == 1) return Convert.ToByte(WIDTH - x - 1);
            return y;
        }

        private byte THIS_W(byte x, byte y)
        {
            /*
                CONNECTION_ANGLE; // угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
                STRIP_DIRECTION;  // направление ленты из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
             */
            if (CONNECTION_ANGLE == 0 && STRIP_DIRECTION == 0) return WIDTH;
            if (CONNECTION_ANGLE == 0 && STRIP_DIRECTION == 1) return HEIGHT;
            if (CONNECTION_ANGLE == 1 && STRIP_DIRECTION == 0) return WIDTH;
            if (CONNECTION_ANGLE == 1 && STRIP_DIRECTION == 3) return HEIGHT;
            if (CONNECTION_ANGLE == 2 && STRIP_DIRECTION == 2) return WIDTH;
            if (CONNECTION_ANGLE == 2 && STRIP_DIRECTION == 3) return HEIGHT;
            if (CONNECTION_ANGLE == 3 && STRIP_DIRECTION == 2) return WIDTH;
            if (CONNECTION_ANGLE == 3 && STRIP_DIRECTION == 1) return HEIGHT;
            return WIDTH;
        }


        private void decodeCutMatrixType(int type)
        {
            /*
                CUT_MATRIX_TYPE;      // тип матрицы: 0 - зигзаг, 1 - параллельная
                CUT_CONNECTION_ANGLE; // угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
                CUT_STRIP_DIRECTION;  // направление ленты из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
             */
            switch (type)
            {
                case 0: // Лево-Верх строки вправо
                    CUT_MATRIX_TYPE = 1;
                    CUT_CONNECTION_ANGLE = 1;
                    CUT_STRIP_DIRECTION = 0;
                    break;
                case 1: // Право-Верх строки влево
                    CUT_MATRIX_TYPE = 1;
                    CUT_CONNECTION_ANGLE = 2;
                    CUT_STRIP_DIRECTION = 2;
                    break;
                case 2: // Лево-Низ строки вправо
                    CUT_MATRIX_TYPE = 1;
                    CUT_CONNECTION_ANGLE = 0;
                    CUT_STRIP_DIRECTION = 0;
                    break;
                case 3: // Право-Низ строки влево
                    CUT_MATRIX_TYPE = 1;
                    CUT_CONNECTION_ANGLE = 3;
                    CUT_STRIP_DIRECTION = 2;
                    break;
                case 4: // Лево-Верх зигзаг вправо
                    CUT_MATRIX_TYPE = 0;
                    CUT_CONNECTION_ANGLE = 1;
                    CUT_STRIP_DIRECTION = 0;
                    break;
                case 5: // Право-Верх зигзаг влево
                    CUT_MATRIX_TYPE = 0;
                    CUT_CONNECTION_ANGLE = 2;
                    CUT_STRIP_DIRECTION = 2;
                    break;
                case 6: // Лево-Низ зигзаг вправо
                    CUT_MATRIX_TYPE = 0;
                    CUT_CONNECTION_ANGLE = 0;
                    CUT_STRIP_DIRECTION = 0;
                    break;
                case 7: // Право-Низ зигзаг влево
                    CUT_MATRIX_TYPE = 0;
                    CUT_CONNECTION_ANGLE = 3;
                    CUT_STRIP_DIRECTION = 2;
                    break;
                case 8: // Лево-Верх колонки вниз
                    CUT_MATRIX_TYPE = 1;
                    CUT_CONNECTION_ANGLE = 1;
                    CUT_STRIP_DIRECTION = 3;
                    break;
                case 9: // Право-Верх колонки вниз
                    CUT_MATRIX_TYPE = 1;
                    CUT_CONNECTION_ANGLE = 2;
                    CUT_STRIP_DIRECTION = 3;
                    break;
                case 10: // Лево-Низ колонки вверх
                    CUT_MATRIX_TYPE = 1;
                    CUT_CONNECTION_ANGLE = 0;
                    CUT_STRIP_DIRECTION = 1;
                    break;
                case 11: // Право-Низ колонки вверх
                    CUT_MATRIX_TYPE = 1;
                    CUT_CONNECTION_ANGLE = 3;
                    CUT_STRIP_DIRECTION = 1;
                    break;
                case 12: // Лево-Верх зигзаг вниз
                    CUT_MATRIX_TYPE = 0;
                    CUT_CONNECTION_ANGLE = 1;
                    CUT_STRIP_DIRECTION = 3;
                    break;
                case 13: // Право-Верх зигзаг вниз
                    CUT_MATRIX_TYPE = 0;
                    CUT_CONNECTION_ANGLE = 2;
                    CUT_STRIP_DIRECTION = 3;
                    break;
                case 14: // Лево-Низ зигзаг вверх
                    CUT_MATRIX_TYPE = 0;
                    CUT_CONNECTION_ANGLE = 0;
                    CUT_STRIP_DIRECTION = 1;
                    break;
                case 15: // Право-Низ зигзаг вверх
                    CUT_MATRIX_TYPE = 0;
                    CUT_CONNECTION_ANGLE = 3;
                    CUT_STRIP_DIRECTION = 1;
                    break;
            }
        }

        // получить номер пикселя в ленте по координатам
        private int getCutPixelNumber(byte x, byte y)
        {
            var xx = CUT_THIS_X(x, y);
            var yy = CUT_THIS_Y(x, y);
            var ww = CUT_THIS_W(x, y);

            return (yy % 2 == 0 || CUT_MATRIX_TYPE == 1)
                ? yy * ww + xx           // если чётная строка
                : yy * ww + ww - xx - 1; // если нечётная строка
        }

        private byte CUT_THIS_X(byte x, byte y)
        {
            /*
                CUT_CONNECTION_ANGLE; // угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
                CUT_STRIP_DIRECTION;  // направление ленты из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
             */
            if (CUT_CONNECTION_ANGLE == 0 && CUT_STRIP_DIRECTION == 0) return x;
            if (CUT_CONNECTION_ANGLE == 0 && CUT_STRIP_DIRECTION == 1) return y;
            if (CUT_CONNECTION_ANGLE == 1 && CUT_STRIP_DIRECTION == 0) return x;
            if (CUT_CONNECTION_ANGLE == 1 && CUT_STRIP_DIRECTION == 3) return Convert.ToByte(CUT_HEIGHT - y - 1);
            if (CUT_CONNECTION_ANGLE == 2 && CUT_STRIP_DIRECTION == 2) return Convert.ToByte(CUT_WIDTH - x - 1);
            if (CUT_CONNECTION_ANGLE == 2 && CUT_STRIP_DIRECTION == 3) return Convert.ToByte(CUT_HEIGHT - y - 1);
            if (CUT_CONNECTION_ANGLE == 3 && CUT_STRIP_DIRECTION == 2) return Convert.ToByte(CUT_WIDTH - x - 1);
            if (CUT_CONNECTION_ANGLE == 3 && CUT_STRIP_DIRECTION == 1) return y;
            return x;
        }

        private byte CUT_THIS_Y(byte x, byte y)
        {
            /*
                CUT_CONNECTION_ANGLE; // угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
                CUT_STRIP_DIRECTION;  // направление ленты из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
             */
            if (CUT_CONNECTION_ANGLE == 0 && CUT_STRIP_DIRECTION == 0) return y;
            if (CUT_CONNECTION_ANGLE == 0 && CUT_STRIP_DIRECTION == 1) return x;
            if (CUT_CONNECTION_ANGLE == 1 && CUT_STRIP_DIRECTION == 0) return Convert.ToByte(CUT_HEIGHT - y - 1);
            if (CUT_CONNECTION_ANGLE == 1 && CUT_STRIP_DIRECTION == 3) return x;
            if (CUT_CONNECTION_ANGLE == 2 && CUT_STRIP_DIRECTION == 2) return Convert.ToByte(CUT_HEIGHT - y - 1);
            if (CUT_CONNECTION_ANGLE == 2 && CUT_STRIP_DIRECTION == 3) return Convert.ToByte(CUT_WIDTH - x - 1);
            if (CUT_CONNECTION_ANGLE == 3 && CUT_STRIP_DIRECTION == 2) return y;
            if (CUT_CONNECTION_ANGLE == 3 && CUT_STRIP_DIRECTION == 1) return Convert.ToByte(CUT_WIDTH - x - 1);
            return y;
        }

        private byte CUT_THIS_W(byte x, byte y)
        {
            /*
                CUT_CONNECTION_ANGLE; // угол подключения: 0 - левый нижний, 1 - левый верхний, 2 - правый верхний, 3 - правый нижний
                CUT_STRIP_DIRECTION;  // направление ленты из угла: 0 - вправо, 1 - вверх, 2 - влево, 3 - вниз
             */
            if (CUT_CONNECTION_ANGLE == 0 && CUT_STRIP_DIRECTION == 0) return CUT_WIDTH;
            if (CUT_CONNECTION_ANGLE == 0 && CUT_STRIP_DIRECTION == 1) return CUT_HEIGHT;
            if (CUT_CONNECTION_ANGLE == 1 && CUT_STRIP_DIRECTION == 0) return CUT_WIDTH;
            if (CUT_CONNECTION_ANGLE == 1 && CUT_STRIP_DIRECTION == 3) return CUT_HEIGHT;
            if (CUT_CONNECTION_ANGLE == 2 && CUT_STRIP_DIRECTION == 2) return CUT_WIDTH;
            if (CUT_CONNECTION_ANGLE == 2 && CUT_STRIP_DIRECTION == 3) return CUT_HEIGHT;
            if (CUT_CONNECTION_ANGLE == 3 && CUT_STRIP_DIRECTION == 2) return CUT_WIDTH;
            if (CUT_CONNECTION_ANGLE == 3 && CUT_STRIP_DIRECTION == 1) return CUT_HEIGHT;
            return CUT_WIDTH;
        }

        private Color getColor(byte b1, byte b2, byte b3, byte order)
        {
            switch (order)
            {
                case 0:  return Color.FromArgb(b1, b2, b3); // RGB
                case 1:  return Color.FromArgb(b1, b3, b2); // RBG
                case 2:  return Color.FromArgb(b3, b2, b1); // BGR
                case 3:  return Color.FromArgb(b3, b2, b1); // BRG
                case 4:  return Color.FromArgb(b2, b3, b1); // GBR
                case 5:  return Color.FromArgb(b2, b1, b3); // GRB
                default: return Color.FromArgb(b1, b2, b3);
            }
        }

        private byte[] getBytes(Color color, byte order)
        {
            var b = new byte[3];
            switch (order)
            {
                case 0:  b[0] = color.R; b[1] = color.G; b[2] = color.B; break; // RGB
                case 1:  b[0] = color.R; b[1] = color.B; b[2] = color.G; break; // RBG
                case 2:  b[0] = color.B; b[1] = color.G; b[2] = color.R; break; // BGR
                case 3:  b[0] = color.B; b[1] = color.R; b[2] = color.G; break; // BRG
                case 4:  b[0] = color.G; b[1] = color.B; b[2] = color.R; break; // GBR
                case 5:  b[0] = color.G; b[1] = color.R; b[2] = color.B; break; // GRB
                default: b[0] = color.R; b[1] = color.G; b[2] = color.B; break; 
            }
            return b;
        }

        // Отрисовка кадра на панели
        private void paintFrame()
        {
            if (frame_buffer == null || frame_buffer.Length == 0) return;

            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => paintFrame())); } catch { }
                return;
            }
            
            // Левый верхний угол кадра в координатах окна, у которого угол x=0, y=0 - левый верхний
            // Левый верхний угол матрицы, у которого угол x=0, y=0 - левый нижний
            var fx = margin + Convert.ToInt32((pnlCanvas.Width - (2 * margin) - frame_w) / 2.0F);
            var fy = margin + Convert.ToInt32((pnlCanvas.Height - (2 * margin) - frame_h) / 2.0F);

            using (var g = Graphics.FromImage(buffer))
            {
                Color color;
                for (byte x = 0; x < WIDTH; x++)
                {
                    for (byte y = 0; y < HEIGHT; y++)
                    {
                        if (file_type == FileType.MOVIE)
                        {
                            var idx = getPixelNumber(x, y) * 3; // 3 байта на цвет
                            if (idx < frame_buffer.Length)
                            {
                                var b1 = frame_buffer[idx];
                                var b2 = frame_buffer[idx + 1];
                                var b3 = frame_buffer[idx + 2];
                                var ix = COLOR_ORDER;
                                color = getColor(b1, b2, b3, ix);
                            }
                            else
                            {
                                color = Color.Black;
                            }
                        }
                        else
                        {
                            // Для файла типа "Картинка" байты в буфере сверху вниз, слева направо, цвет - RGB
                            var idx = (x * HEIGHT + y) * 3;
                            if (idx < frame_buffer.Length)
                            {
                                var cr = frame_buffer[idx];
                                var cg = frame_buffer[idx + 1];
                                var cb = frame_buffer[idx + 2];
                                color = Color.FromArgb(cr, cg, cb);
                            }
                            else
                            {
                                color = Color.Black;
                            }
                        }

                        using (var brush = new SolidBrush(color))
                        {
                            var px = fx + x * pix_size;
                            var py = fy + frame_h - (y + 1) * pix_size;
                            g.FillRectangle(brush, px, py, pix_size, pix_size);
                        }
                    }
                }

                // Рисовать границы кадра
                if (!(playing || save_in_porcess))
                {
                    var pen = new Pen(Color.Gray);
                    g.DrawRectangle(pen, fx, fy, frame_w, frame_h);
                }

                // Рисовать сетку
                if (chkGrid.Checked)
                {
                    var pen = new Pen(Color.Gray);
                    for (int i = 1; i < edtWidth.Value; i++)
                    {
                        var ix = fx + i * pix_size;
                        g.DrawLine(pen, ix, fy, ix, fy + frame_h);
                    }

                    for (int i = 1; i < edtHeight.Value; i++)
                    {
                        var iy = fy + i * pix_size;
                        g.DrawLine(pen, fx, iy, fx + frame_w, iy);
                    }
                }

                // Если включена обрезка - рисовать рамку обрезаемой площади
                if (chkCutFrame.Checked && file_type == FileType.MOVIE)
                {
                    // Координаты поля рисования - 0,0 - верхний левый угол холста
                    // Координаты 0,0 матрицы - левый нижний угол окна (поля матрицы)
                    var dy = (int)(edtCutOffsetY.Maximum - edtCutOffsetY.Value);
                    var pen = new Pen(Color.Lime);
                    var ix  = fx + (int)edtCutOffsetX.Value * pix_size;
                    var iy  = fy + dy * pix_size;
                    var ix2 = fx + (int)(edtCutOffsetX.Value + edtCutWidth.Value) * pix_size;
                    var iy2 = fy + (int)(dy + edtCutHeight.Value) * pix_size;
                    g.DrawLine(pen, ix,  iy,  ix2, iy);
                    g.DrawLine(pen, ix,  iy,  ix,  iy2);
                    g.DrawLine(pen, ix2, iy,  ix2, iy2);
                    g.DrawLine(pen, ix,  iy2, ix2, iy2);
                }
            }

            pictureBox.Image = buffer;
        }

        void openDataFile(string fname)
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => openDataFile(fname))); } catch { }
                return;
            }

            var frame_ok = openDataFile2(fname);

            if (frame_ok)
            {
                changing = true;
                lblCurrentFrame.Text = "1";
                lblTotalFrames.Text = frames_num.ToString();

                trkFrames.Minimum = 0;
                trkFrames.Maximum = frames_num - 1;
                trkFrames.Value = 0;

                edtFragStart.Minimum = trkFrames.Minimum + 1;
                edtFragStart.Maximum = trkFrames.Maximum + 1;
                edtFragStart.Value = trkFrames.Minimum + 1;

                edtFragEnd.Minimum = trkFrames.Minimum + 1;
                edtFragEnd.Maximum = trkFrames.Maximum + 1;
                edtFragEnd.Value = trkFrames.Maximum + 1;
                changing = false;

                if (!(playing || save_in_porcess))
                {
                    btnStart.Enabled = true;
                    lblFirstFrame.Visible = true;
                    lblCurrentFrame.Visible = true;
                    lblTotalFrames.Visible = true;
                    trkFrames.Enabled = true;
                    edtFragStart.Enabled = true;
                    edtFragEnd.Enabled = true;
                    btnFragStart.Enabled = true;
                    btnFragEnd.Enabled = true;
                    btnFragSave.Enabled = true;
                }
            }
            else
            {
                using (new CenterWinDialog(this))
                {
                    MessageBox.Show(Strings.FileOpenError + " '" + fname + "'\n\n" + lastError, Strings.Error, MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }

            pictureBox.Refresh();
        }

        bool openDataFile2(string fname)
        {
            var frame_ok = true;
            try
            {
                if (file != null) file.Close();
                file = File.Open(fname, FileMode.Open);
                file_name = fname;
                file_type = Path.GetExtension(fname).ToLower() == ".p" ? FileType.PICTURE : FileType.MOVIE;

                if (file.Length == 0)
                {
                    throw new Exception(Strings.NoDataInFile);
                }

                frames_num = file_type == FileType.MOVIE ? Convert.ToInt32(file.Length / (frame_s + 1)) : 1;
                frame_buffer = LoadFrameFromFile(0);
                frame_ok = frame_buffer.Length > 0;
            }
            catch (Exception ex)
            {
                lastError = ex.Message;
                frame_ok = false;
            }
            return frame_ok;
        }

        /// <summary>
        /// Выбор папки
        /// </summary>
        /// <param name="DataPath">Папка по умолчанию</param>
        /// <param name="Title">Заголовок окна диалога выбора папки</param>
        /// <returns>Папка хранилища</returns>
        public string SelectDataFolder(string DataPath, string Title = "")
        {
            var folders = SelectDataFolders(DataPath, false, Title);

            var dataFolder = folders.Length > 0 ? folders[0] : "";

            if (string.IsNullOrWhiteSpace(dataFolder)) return "";

            return dataFolder;
        }

        /// <summary>
        /// Множественный выбор папкок
        /// </summary>
        /// <param name="DataPath">Папка по умолчанию</param>
        /// <param name="multiSelect">Флаг: допускается выбор нескольких папок</param>
        /// <param name="Title">Заголовок диалога выбора папки</param>
        /// <returns>Список папок</returns>
        public string[] SelectDataFolders(string DataPath, bool multiSelect = false, string Title = "")
        {
            var dialog = new FolderPicker
            {
                MultiSelect = multiSelect,
                InitialDirectory = DataPath,
                Title = string.IsNullOrWhiteSpace(Title) ? Strings.ChooseFolder : Title,
            };

            if (!dialog.ShowDialog()) return new string[0];

            return dialog.FileNames;
        }

        void ScanFiles(string folder)
        {
            if (string.IsNullOrWhiteSpace(folder) || !Directory.Exists(folder))
            {
                folder = Path.GetDirectoryName(Application.ExecutablePath);
            }

            var save_changing = changing;
            changing = true;
            listFiles.Items.Clear();

            var list = new List<string>();
            list.AddRange(Directory.GetFiles(folder, "*.out", SearchOption.TopDirectoryOnly));
            list.AddRange(Directory.GetFiles(folder, "*.p", SearchOption.TopDirectoryOnly));
            list.Sort();

            if (list.Count > 0)
            {
                foreach (var file in list)
                {
                    listFiles.Items.Add(new FileListBoxItem(Path.GetFileName(file)));
                }
            }

            changing = save_changing;
        }

        void SetTitle()
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => SetTitle())); } catch { }
                return;
            }

            var text = "Jinx Frame Viewer";
            if (playing || save_in_porcess)
            {
                if (file_index >= 0)
                {
                    var item = (FileListBoxItem)listFiles.Items[file_index];
                    text = String.Format("Jinx Frame Viewer [{0}]", item.FullName);
                }
            }
            else
            {
                if (listFiles.SelectedIndex >= 0)
                {
                    var item = (FileListBoxItem)listFiles.Items[listFiles.SelectedIndex];
                    text = String.Format("Jinx Frame Viewer [{0}]", item.FullName);
                }
            }

            Text = text;
        }

        void CheckCutDimensions()
        {
            if (changing) return;

            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => CheckCutDimensions())); } catch { }
                return;
            }

            pnlCutFrame.Enabled = chkCutFrame.Checked;

            if (edtCutWidth.Value > edtWidth.Value) edtCutWidth.Value = edtWidth.Value;
            edtCutWidth.Minimum = edtWidth.Minimum;
            edtCutWidth.Maximum = edtWidth.Maximum;

            if (edtCutHeight.Value > edtHeight.Value) edtCutHeight.Value = edtHeight.Value;
            edtCutHeight.Minimum = edtHeight.Minimum;
            edtCutHeight.Maximum = edtHeight.Maximum;

            var val_x = edtCutOffsetX.Value;
            while (edtCutWidth.Value + val_x > edtWidth.Value && val_x > 0) val_x--;
            edtCutOffsetX.Value = val_x;

            var val_y = edtCutOffsetY.Value;
            while (edtCutHeight.Value + val_y > edtHeight.Value && val_y > 0) val_y--;
            edtCutOffsetY.Value = val_y;

            edtCutOffsetX.Maximum = edtWidth.Value - edtCutWidth.Value;
            edtCutOffsetY.Maximum = edtHeight.Value - edtCutHeight.Value;

            CUT_WIDTH = Convert.ToByte(edtCutWidth.Value);
            CUT_HEIGHT = Convert.ToByte(edtCutHeight.Value);
            CUT_OFFSET_X = Convert.ToByte(edtCutOffsetX.Value);
            CUT_OFFSET_Y = Convert.ToByte(edtCutOffsetY.Value);
            CUT_COLOR_ORDER = (byte)cbCutColorOrder.SelectedIndex;

            pictureBox.Refresh();
        }

        private void disableControls()
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => disableControls())); } catch { }
                return;
            }

            btnStart.Enabled = false;
            btnStop.Enabled = true;
            btnOpenFile.Enabled = false;
            edtWidth.Enabled = false;
            edtHeight.Enabled = false;
            listFiles.Enabled = false;
            btnFragStart.Enabled = false;
            btnFragEnd.Enabled = false;
            edtFragStart.Enabled = false;
            edtFragEnd.Enabled = false;
            chkCutFrame.Enabled = false;
            btnStartCut.Enabled = false;
            btnFragSave.Enabled = false;
            btnFrameSave.Enabled = false;
            btnFragCodeSave.Enabled = false;
            btnFrameCodeSave.Enabled = false;
            btnPrevFrame.Enabled = false;
            btnNextFrame.Enabled = false;
            btnDelete.Enabled = false;

            pnlMatrixType.Enabled = false;
            pnlCreate.Enabled = false;
            pnlAddFrame.Enabled = false;
            pnlFragmentRange.Enabled = false;
            pnlFragmentSave.Enabled = false;
            pnlCutFrame.Enabled = false;

            btnStartEdit.Enabled = false;
            btnSaveEdit.Enabled = false;
            btnCancelEdit.Enabled = false;
            colorEditor.Enabled = false;
            colorWheel.Enabled = false;
            screenColorPickerFore.Enabled = false;
            screenColorPickerBack.Enabled = false;
            btnFill.Enabled = false;
            btnCopy.Enabled = false;
            btnPaste.Enabled = false;
            btnSetActivePickerFore.Enabled = false;
            btnSetActivePickerBack.Enabled = false;
            colorPalette.Enabled = false;
        }

        private void enableControls()
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => enableControls())); } catch { }
                return;
            }

            var enable = listFiles.Items.Count > 0;
            var enable2 = enable && !playing;
            var enable3 = enable2 && !inEditMode;
            var isPicture = file_type == FileType.PICTURE;

            btnStart.Enabled = enable3;
            btnStop.Enabled = playing;
            btnOpenFile.Enabled = !(playing || inEditMode);
            edtWidth.Enabled  = enable3;
            edtHeight.Enabled = enable3;

            pnlMatrixType.Enabled = enable3 && !isPicture;
            pnlSpeed.Enabled = enable3 && !isPicture;
            pnlCut.Enabled = enable3 && !isPicture;
            pnlFragmentRange.Enabled = enable3 && !isPicture;
            pnlAddFrame.Enabled = enable3 && !isPicture;
            pnlCutButton.Enabled = enable3 && !isPicture;
            pnlFragmentSave.Enabled = enable3;
            pnlCutFrame.Enabled = enable3 && chkCutFrame.Checked;
            pnlCreate.Enabled = !(playing || inEditMode);
            btnDelete.Enabled = enable3;

            pnlMatrixType.Visible = enable2 && !isPicture;
            pnlSpeed.Visible = enable2 && !isPicture;
            pnlCut.Visible = enable2 && !isPicture;
            pnlFragmentRange.Visible = enable2 && !isPicture;
            pnlAddFrame.Visible = enable2 && !isPicture;
            pnlCutButton.Visible = enable2 && !isPicture;

            listFiles.Enabled = enable3;
            btnFragStart.Enabled = enable3;
            btnFragEnd.Enabled = enable3;
            edtFragStart.Enabled = enable3;
            edtFragEnd.Enabled = enable3;
            chkCutFrame.Enabled = enable3;
            btnStartCut.Enabled = enable3 && chkCutFrame.Checked;
            trkFrames.Enabled = enable3;
            trkSpeed.Enabled = enable3;
            btnFragSave.Enabled = enable3 && !isPicture;
            btnFrameSave.Enabled = enable3 && !isPicture;
            btnFragCodeSave.Enabled = enable3 && !isPicture;
            btnFrameCodeSave.Enabled = enable3;
            btnPrevFrame.Enabled = enable3;
            btnNextFrame.Enabled = enable3;

            btnStartEdit.Enabled = enable3;
            btnSaveEdit.Enabled = inEditMode && isFrameChanged;
            btnCancelEdit.Enabled = inEditMode;
            colorEditor.Enabled = enable2;
            colorWheel.Enabled = enable2;
            screenColorPickerFore.Enabled = enable2;
            screenColorPickerBack.Enabled = enable2;
            btnFill.Enabled = inEditMode;
            btnCopy.Enabled = enable2;
            btnPaste.Enabled = inEditMode && clipboard_frame_buffer != null && clipboard_frame_buffer.Length == frame_s;
            btnAddFrames.Enabled = enable3 && frame_size_ok;
            btnSetActivePickerFore.Enabled = enable2;
            btnSetActivePickerBack.Enabled = enable2;
            colorPalette.Enabled = enable2;

            cbLanguage.Enabled = !inEditMode;

            lblWarning.Visible = !frame_size_ok && listFiles.Items.Count > 0;
            btnStartCut.Text = Strings.CutAction; // "Обрезать"

            Invalidate();
        }

        private void processCutFiles(string src_folder, string dst_folder, List<string> files)
        {
            initFileProgress(files.Count);
            file_index = 0;

            foreach (var fName in files)
            {
                if (!save_in_porcess) break;

                updateFileProgress(fName);

                try
                {
                    var file_in = Path.Combine(src_folder, fName);
                    var file_out = Path.Combine(dst_folder, fName);

                    openDataFile2(file_in);

                    initFrameProgress();

                    SetTitle();

                    // Сохранить фрагмент в новый временный файл
                    using (var writer = new BinaryWriter(File.Open(file_out, FileMode.Create)))
                    {
                        for (var i = 0; i < frames_num; i++)
                        {
                            updateFrameProgress();

                            frame_buffer = LoadFrameFromFile(i);
                            var ok = frame_buffer.Length > 0;
                            if (!ok) throw new Exception(lastError);
                            RefreshPictureBox();

                            for (byte x = 0; x < CUT_WIDTH; x++)
                            {
                                for (byte y = 0; y < CUT_HEIGHT; y++)
                                {
                                    var idx = getPixelNumber((byte)(CUT_OFFSET_X + x), (byte)(CUT_OFFSET_Y + y)) * 3; // 3 байта на цвет
                                    var b1 = frame_buffer[idx];
                                    var b2 = frame_buffer[idx + 1];
                                    var b3 = frame_buffer[idx + 2];

                                    var color = getColor(b1, b2, b3, COLOR_ORDER);
                                    var arr = getBytes(color, CUT_COLOR_ORDER);

                                    idx = getCutPixelNumber(x, y) * 3; // 3 байта на цвет

                                    cut_frame_buffer[idx] = arr[0];
                                    cut_frame_buffer[idx + 1] = arr[1];
                                    cut_frame_buffer[idx + 2] = arr[2];
                                }
                            }

                            // Разделитель кадров в файле
                            writer.Write(frame_marker);

                            // Сохранить буфер кадра
                            for (var ii = 0; ii < cut_frame_s; ii++)
                            {
                                writer.Write(cut_frame_buffer[ii]);
                            }
                        }

                        writer.Close();
                    }
                }
                catch (Exception ex)
                {
                    lastError = ex.Message;
                }
                finally
                {
                    if (file != null)
                    {
                        file.Close();
                        file = null;
                    }
                }

                file_index++;
            }

            file_name = Path.Combine(src_folder, files[0] + ".out");
            openDataFile2(file_name);

            save_in_porcess = false;
            stopProgress();
        }

        void RefreshPictureBox()
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => RefreshPictureBox())); } catch { }
                return;
            }

            pictureBox.Refresh();
        }

        void initFileProgress(int file_total)
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => initFileProgress(file_total))); } catch { }
                return;
            }

            pnlCutProgress.Visible = true;
            progressFile.Minimum = 0;
            progressFile.Value = 1;
            progressFile.Maximum = file_total;
        }

        void initFrameProgress()
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => initFrameProgress())); } catch { }
                return;
            }

            changing = true;
            lblCurrentFrame.Text = "1";
            lblTotalFrames.Text = frames_num.ToString();
            trkFrames.Minimum = 0;
            trkFrames.Value = 0;
            trkFrames.Maximum = frames_num;
            progressFrame.Minimum = 0;
            progressFrame.Value = 1;
            progressFrame.Maximum = frames_num;
            changing = false;
        }

        void updateFileProgress(string fileName)
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => updateFileProgress(fileName))); } catch { }
                return;
            }

            lblFileInProcess.Text = fileName;
            progressFile.PerformStep();
        }

        void updateFrameProgress()
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => updateFrameProgress())); } catch { } 
                return;
            }

            changing = true;
            progressFrame.PerformStep();
            trkFrames.Value = trkFrames.Value + 1;
            lblCurrentFrame.Text = trkFrames.Value.ToString();
            changing = false;
        }

        void stopProgress()
        {
            if (InvokeRequired)
            {
                try { Invoke((MethodInvoker)(() => stopProgress())); } catch { }
                return;
            }

            pnlCutProgress.Visible = false;

            changing = true;
            lblCurrentFrame.Text = "1";
            lblTotalFrames.Text = frames_num.ToString();
            trkFrames.Minimum = 0;
            trkFrames.Value = 0;
            trkFrames.Maximum = frames_num;
            progressFrame.Minimum = 0;
            progressFrame.Value = 1;
            progressFrame.Maximum = frames_num;
            changing = false;

            enableControls();
        }

        private void cbLanguage_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (changing) return;
            currentLanguage = ((CultureInfo)cbLanguage.SelectedItem).Name;
            lblLanguageWarning.Visible = true;
        }

        private void screenColorPicker_ColorChanged(object sender, EventArgs e)
        {
            if (changing) return;
            var picker = (ScreenColorPicker) sender;
            colorWheel.Color = picker.Color;
            colorEditor.Color = picker.Color;
        }

        private void colorWheel_ColorChanged(object sender, EventArgs e)
        {
            if (changing) return;
            var color = Color.FromArgb(255, colorWheel.Color);
            activePicker.Color = color;
            activePicker.BackColor = color;
            colorEditor.Color = colorWheel.Color;
        }

        private void colorEditor_ColorChanged(object sender, EventArgs e)
        {
            if (changing) return;
            var color = Color.FromArgb(255, colorEditor.Color);
            activePicker.Color = activePicker.BackColor = color;
            colorWheel.Color = colorEditor.Color;
        }

        private void colorPalette_MouseUp(object sender, MouseEventArgs e)
        {
            if (!(e.Button == MouseButtons.Left || e.Button == MouseButtons.Right)) return;

            var color = Color.FromArgb(255, colorPalette.Color);
            changing = true;
            if (e.Button == MouseButtons.Left)
                screenColorPickerFore.Color = screenColorPickerFore.BackColor = color;
            else
                screenColorPickerBack.Color = screenColorPickerBack.BackColor = color;
            colorEditor.Color = color;
            colorWheel.Color = color;
            changing = false;
        }

        private void screenColorPicker_MouseUp(object sender, MouseEventArgs e)
        {
            var picker = (ScreenColorPicker)sender;
            picker.BackColor = picker.Color;
            picker.ClearSnapshotImage();
        }

        private void screenColorPickerFore_MouseDown(object sender, MouseEventArgs e)
        {
            var picker = (ScreenColorPicker)sender;
            SetActivePicker(picker);
        }

        private void btnStartEdit_Click(object sender, EventArgs e)
        {
            inEditMode = true;
            pictureBox.Cursor = penCursor;
            enableControls();
        }

        private void btnSaveEdit_Click(object sender, EventArgs e)
        {
            inEditMode = false;
            pictureBox.Cursor = Cursors.Default;
            if (isFrameChanged)
            {
                // Сохранить текущий кадр
                SaveFrameToFile(trkFrames.Value);
                isFrameChanged = false;
            }
            enableControls();
        }

        private void btnCancelEdit_Click(object sender, EventArgs e)
        {
            inEditMode = false;
            pictureBox.Cursor = Cursors.Default;
            if (isFrameChanged)
            {
                // Перезагрузить текущий кадр
                frame_buffer = LoadFrameFromFile(trkFrames.Value);
                isFrameChanged = false;
            }
            enableControls();
        }

        private void pictureBox_MouseClick(object sender, MouseEventArgs e)
        {
            if (!inEditMode) return;
            if (!(e.Button == MouseButtons.Left || e.Button == MouseButtons.Right)) return;

            // Левый верхний угол кадра в координатах окна, у которого угол x=0, y=0 - левый верхний
            // Левый верхний угол матрицы, у которого угол x=0, y=0 - левый нижний
            var fx = Convert.ToInt32((pnlCanvas.Width - 2 * margin - frame_w) / 2.0F) + margin;
            var fy = Convert.ToInt32((pnlCanvas.Height - 2 * margin - frame_h) / 2.0F) + margin;

            var x = Convert.ToInt32((e.X - fx - pix_size / 2) / pix_size);
            var y = HEIGHT - 1 - Convert.ToInt32((e.Y - fy - pix_size / 2) / pix_size);

            if (x >= WIDTH || x < 0 || y >= HEIGHT || y < 0) return;

            byte[] arr = new byte[3];
            int idx;

            var picker = e.Button == MouseButtons.Right ? screenColorPickerBack : screenColorPickerFore;

            if (file_type == FileType.MOVIE)
            {
                arr = getBytes(picker.Color, COLOR_ORDER);
                idx = getPixelNumber(Convert.ToByte(x), Convert.ToByte(y)) * 3; // 3 байта на цвет
            }
            else
            {
                arr[0] = picker.Color.R;
                arr[1] = picker.Color.G;
                arr[2] = picker.Color.B;
                idx = (x * HEIGHT + y) * 3;
            }

            frame_buffer[idx] = arr[0];
            frame_buffer[idx + 1] = arr[1];
            frame_buffer[idx + 2] = arr[2];

            isFrameChanged = true;
            btnSaveEdit.Enabled = true;

            pictureBox.Refresh();
        }

        private void btnPrevFrame_Click(object sender, EventArgs e)
        {
            if (trkFrames.Value > 0) trkFrames.Value -= 1;
        }

        private void btnNextFrame_Click(object sender, EventArgs e)
        {
            if (trkFrames.Value < frames_num - 1) trkFrames.Value += 1;
        }

        private void SetActivePicker(ScreenColorPicker picker)
        {
            activePicker = picker;
            if (activePicker == screenColorPickerFore)
            {
                btnSetActivePickerFore.FlatAppearance.BorderColor = Color.Navy;
                btnSetActivePickerFore.BackColor = Color.SkyBlue;
                btnSetActivePickerBack.FlatAppearance.BorderColor = Color.Silver;
                btnSetActivePickerBack.BackColor = SystemColors.Control;
            }
            else
            {
                btnSetActivePickerFore.FlatAppearance.BorderColor = Color.Silver;
                btnSetActivePickerFore.BackColor = SystemColors.Control;
                btnSetActivePickerBack.FlatAppearance.BorderColor = Color.Navy;
                btnSetActivePickerBack.BackColor = Color.SkyBlue;
            }
        }

        private void btnSetActivePickerFore_Click(object sender, EventArgs e)
        {
            SetActivePicker(sender == btnSetActivePickerFore ? screenColorPickerFore : screenColorPickerBack);
        }

        private void btnFill_Click(object sender, EventArgs e)
        {
            var color = activePicker.Color;
            var bytes = getBytes(color, COLOR_ORDER);

            for (byte x = 0; x < WIDTH; x++)
            {
                for (byte y = 0; y < HEIGHT; y++)
                {
                    var idx = getPixelNumber(x, y) * 3; // 3 байта на цвет
                    frame_buffer[idx]     = bytes[0];
                    frame_buffer[idx + 1] = bytes[1];
                    frame_buffer[idx + 2] = bytes[2];
                }
            }

            isFrameChanged = true;
            enableControls();

            pictureBox.Invalidate();
        }

        private void btnCreateNew_Click(object sender, EventArgs e)
        {
            var ok = true;
            var create_type = rbMovie.Checked ? FileType.MOVIE : FileType.PICTURE;
            var new_file_name = edtCreateNewFileName.Text.Trim();
            if (new_file_name.Length == 0)
            {
                using (new CenterWinDialog(this))
                {
                    MessageBox.Show(create_type == FileType.MOVIE ? Strings.EmptyFileNameMovie : Strings.EmptyFileNamePicture, Strings.Error, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }
            }

            var file_ext = create_type == FileType.MOVIE ? ".out" : ".p";
            if (!new_file_name.ToLower().EndsWith(file_ext)) new_file_name += file_ext;

            var invalidChars = string.Join("", Path.GetInvalidFileNameChars());
            var regex = new Regex("[" + Regex.Escape(string.Join("", invalidChars)) + "]");

            if (regex.IsMatch(new_file_name))
            {
                using (new CenterWinDialog(this))
                {
                    MessageBox.Show(Strings.InvalidFileName, Strings.Error, MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                return;
            }

            FileStream new_file = null;
            try
            {
                var full_file_name = Path.Combine(src_folder, new_file_name);
                if (File.Exists(full_file_name))
                {
                    using (new CenterWinDialog(this))
                    {
                        var res = MessageBox.Show(Strings.FileExists + " '" + new_file_name + "'\n\n" + Strings.Overwrite, Strings.Error, MessageBoxButtons.OKCancel, MessageBoxIcon.Question);
                        if (res != DialogResult.OK) return;
                    }                
                }

                frame_s = WIDTH * HEIGHT * 3;
                frame_buffer = new byte[frame_s];
                Array.Clear(frame_buffer, 0, frame_s);

                new_file = File.Open(full_file_name, FileMode.CreateNew, FileAccess.Write);

                if (create_type == FileType.MOVIE)
                {
                    var fr_num = Convert.ToInt32(edtCreateNewFrames.Value);
                    if (frame_marker == 0) frame_marker = 1;

                    // Создать новый файл ролика
                    using (var writer = new BinaryWriter(new_file))
                    {
                        for (int n = 0; n < fr_num; n++)
                        {
                            // Сохранить метку начала кадра
                            writer.Write(frame_marker);
                            // Сохранить буфер кадра
                            for (var i = 0; i < frame_s; i++)
                            {
                                writer.Write(frame_buffer[i]);
                            }
                        }
                        writer.Close();
                    }
                }
                else
                {
                    // Создать новый файл картинки
                    using (var writer = new BinaryWriter(new_file))
                    {
                        // Сохранить метку начала кадра
                        writer.Write((byte)0x33);
                        writer.Write(WIDTH);
                        writer.Write(HEIGHT);
                        // Сохранить буфер кадра
                        for (var i = 0; i < frame_s; i++)
                        {
                            writer.Write(frame_buffer[i]);
                        }
                        writer.Close();
                    }
                }
            }
            catch (Exception ex)
            {
                ok = false;
                lastError = ex.Message;
            }
            finally
            {
                if (new_file != null) new_file.Close();
            }

            if (!ok)
            {
                using (new CenterWinDialog(this))
                {
                    MessageBox.Show((file_type == FileType.MOVIE ? Strings.CreateMovieError : Strings.CreatePictureError) + "\n'" + new_file_name + "'\n\n" + lastError, 
                                    Strings.Error, MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                return;
            }

            changing = true;
            listFiles.SelectedItems.Clear();
            changing = false;

            var item = new FileListBoxItem(new_file_name);
            listFiles.Items.Add(item);
            listFiles.SelectedItem = item;
        }

        private void btmAddFrames_Click(object sender, EventArgs e)
        {
            var ok = true;
            var go_to_frame = frames_num;
            try
            {
                if (file != null) file.Close();
                
                // Сохранить фрагмент в новый временный файл
                file = File.Open(file_name, FileMode.Append, FileAccess.Write);
                using (var writer = new BinaryWriter(file))
                {
                    Array.Clear(frame_buffer, 0, frame_s);

                    file.Position = file.Length;
                    var cnt = Convert.ToInt32(edtAddFrames.Value);
                    for (int n = 0; n < cnt; n++)
                    {
                        // Сохранить метку начала кадра
                        writer.Write(frame_marker);
                        // Сохранить буфер кадра
                        for (var i = 0; i < frame_s; i++)
                        {
                            writer.Write(frame_buffer[i]);
                        }
                    }
                    writer.Close();
                }
            }
            catch (Exception ex)
            {
                ok = false;
                lastError = ex.Message;
            }
            finally
            {
                if (file != null)
                {
                    file.Close();
                    file = null;
                }
            }

            // Reopen file
            ok &= openDataFile2(file_name);
            if (ok)
            {
                initFrameProgress();
                trkFrames.Value = go_to_frame;
            }
        }

        private void btnCopy_Click(object sender, EventArgs e)
        {
            clipboard_frame_buffer = new byte[frame_s];
            for (int i = 0; i < frame_s; i++)
            {
                clipboard_frame_buffer[i] = frame_buffer[i];
            }
            enableControls();
        }

        private void btnPaste_Click(object sender, EventArgs e)
        {
            for (int i = 0; i < frame_s; i++)
            {
                frame_buffer[i] = clipboard_frame_buffer[i];
            }
            isFrameChanged = true;
            enableControls();
        }

        private void listFiles_DrawItem(object sender, DrawItemEventArgs e)
        {
            if (e.Index < 0) return;

            var item = (FileListBoxItem)listFiles.Items[e.Index];
            var y = e.Index * listFiles.ItemHeight;
            var backColor = e.State.HasFlag(DrawItemState.Selected) ? Color.SkyBlue : e.BackColor;
            using (SolidBrush textBrush = new SolidBrush(item.ItemColor), backBrush = new SolidBrush(backColor)) {
                e.Graphics.FillRectangle(backBrush, e.Bounds);
                e.Graphics.DrawString(item.Name, listFiles.Font, textBrush, 0, e.Bounds.Top);
                e.Graphics.DrawString(item.Extension, listFiles.Font, textBrush, listFiles.Width - 44, e.Bounds.Top);
            }
        }

        private void btnDelete_Click(object sender, EventArgs e)
        {
            using (new CenterWinDialog(this))
            {
                var res = MessageBox.Show(Strings.DeleteFiles, Strings.Error, MessageBoxButtons.OKCancel, MessageBoxIcon.Question);
                if (res != DialogResult.OK) return;
            }

            if (listFiles.SelectedItems.Count > 0)
            {
                var save_idx = listFiles.SelectedIndex;
                var sb = new StringBuilder();
                if (file != null) file.Close();
                foreach (var itm in listFiles.SelectedItems)
                {
                    var item = (FileListBoxItem)itm;
                    var full_name = Path.Combine(src_folder, item.FullName);
                    try
                    {
                        File.Delete(full_name);
                    }
                    catch (Exception ex)
                    {
                        sb.AppendLine("  '" + item.FullName + "'");
                    }
                }

                if (sb.Length > 0)
                {
                    using (new CenterWinDialog(this))
                    {
                        MessageBox.Show(Strings.FileDeleteError + "\n" + sb, Strings.Error, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }

                changing = true;
                ScanFiles(src_folder);
                changing = false;
                if (save_idx < 0 || save_idx >= listFiles.Items.Count) save_idx = 0;
                listFiles.SelectedIndex = save_idx;

            }
        }
    }

    public class FileListBoxItem
    {
        public FileListBoxItem(string file, byte width = 0, byte height = 0)
        {
            Name = Path.GetFileNameWithoutExtension(file);
            Extension = Path.GetExtension(file) ?? "";
            ItemColor = Extension.ToLower() == ".p" ? Color.Green : Color.Navy;
            Type = Extension.ToLower() == ".p" ? FileType.PICTURE : FileType.MOVIE;
            Width = width;
            Height = height;
        }
        public FileType Type { get; private set; }
        public Color ItemColor { get; private set; }
        public string Name { get; private set; }
        public string Extension { get; private set; }
        public byte Width { get; set; }
        public byte Height { get; set; }
        public string FullName
        {
            get { return Name + Extension; }
        }
        public override string ToString()
        {
            return FullName;
        }
    }

}
