﻿using System;
using System.Drawing;
using System.IO;
using System.Reflection;
using System.Text;
using System.Windows.Forms;
using System.Xml;

namespace FontMaker
{
    public partial class Form1 : Form
    {
        private uint[] uhex;               // Битовый массив точек 32x32 - uhex[0] - крайний левый столбец точек, b0 (младший бит) - верхняя точка (левый верхний угол изображения)
                                           // для матриц меньшего размера чем 32x32 используется соответствующее кол-во бит в слове, и первые элементы массива

        private string opt_file;           // Имя файла для хранения текущих настроек

        private byte charWidth;            // Ширина и высота рисуемого символа
        private byte charHeight;

        private Point pointLeftUp;         // Точка отсчета - левый верхний угол изображения, выводимого в области рисования
        private byte pnlSize;              // Размер стороны квадрата "точки" рисунка в пикселях

        private bool processing;

        private readonly byte margin = 6;  // Отступы для рисования в поле, чтобы линии не "прилипали" к границам

        public Form1()
        {
            InitializeComponent();
            opt_file = Path.ChangeExtension(Application.ExecutablePath, ".opt");

            // Битовый массив точек 32x32
            uhex = new uint[32];
            pointLeftUp = new Point();

            charWidth = Convert.ToByte(edtFontWidth.Value);
            charHeight = Convert.ToByte(edtFontHeight.Value);

            typeof(Panel).InvokeMember("DoubleBuffered", BindingFlags.SetProperty | BindingFlags.Instance | BindingFlags.NonPublic, null, pnlCharImage, new object[] { true });
            typeof(Panel).InvokeMember("DoubleBuffered", BindingFlags.SetProperty | BindingFlags.Instance | BindingFlags.NonPublic, null, pnlHexLines, new object[] { true });

            if (File.Exists(opt_file))
            {
                try
                {
                    int l = Left, t = Top, w = Width, h = Height;
                    using (BinaryReader reader = new BinaryReader(File.Open(opt_file, FileMode.Open)))
                    {
                        // Загрузить настройки и поле ресунка, сохраненные в предыдущем сеансе
                        edtFontWidth.Value = charWidth = reader.ReadByte();
                        edtFontHeight.Value = charHeight = reader.ReadByte();
                        for (var i = 0; i < 32; i++) uhex[i] = reader.ReadUInt32();
                        t = reader.ReadInt32();
                        l = reader.ReadInt32();
                        w = reader.ReadInt32();
                        h = reader.ReadInt32();
                        CheckWindowPosition(ref l, ref t, ref w, ref h);
                        StartPosition = FormStartPosition.Manual;
                        Left = l;
                        Top = t;
                        Width = w;
                        Height = h;
                    }
                }
                catch { }
            }

        }

        private void Form1_Load(object sender, EventArgs e)
        {
            updateAll();
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            // создаем объект BinaryWriter
            try
            {
                using (var writer = new BinaryWriter(File.Open(opt_file, FileMode.Create)))
                {
                    // Записываем в файл значение каждого поля структуры
                    writer.Write(Convert.ToByte(edtFontWidth.Value));
                    writer.Write(Convert.ToByte(edtFontHeight.Value));
                    for (var i = 0; i < 32; i++) writer.Write(Mask(uhex[i], charHeight));
                    writer.Write(Top);
                    writer.Write(Left);
                    writer.Write(Width);
                    writer.Write(Height);
                }
            }
            catch { }
        }

        private void charSizeChanged(object sender, EventArgs e)
        {
            // Изменили размерность поля - пересоздать и перерисовать всё
            charWidth = Convert.ToByte(edtFontWidth.Value);
            charHeight = Convert.ToByte(edtFontHeight.Value);
            updateAll();
        }

        private void btnClear_Click(object sender, EventArgs e)
        {
            for (var i = 0; i < 32; i++) uhex[i] = 0;
            updateAll();
        }

        private void pnlCharImage_Resize(object sender, EventArgs e)
        {
            pnlCharImage.Invalidate();
            pnlHexLines.Invalidate();
        }

        private void pnlCharImage_MouseDown(object sender, MouseEventArgs e)
        {
            // Поставить/Убрать точку в поле, куда ткнули мышкой, изменить значение в массиве и все перерисовать
            var x = e.Location.X - pointLeftUp.X;
            var y = e.Location.Y - pointLeftUp.Y;
            if (x < 0 || y < 0 || x > charWidth * pnlSize || y > charHeight * pnlSize)
            {
                updateHexArray();
                return;
            }

            var px = Convert.ToByte(x / pnlSize);
            var py = Convert.ToByte(y / pnlSize);

            var val = uhex[px];
            var fill = (val & (0x01 << py)) > 0;
            if (fill)
                val = val & (uint)(~(0x01 << py));
            else
                val = val | (uint)(0x01 << py);

            uhex[px] = val;
            updateAll();
        }

        private void edtHexArray_KeyDown(object sender, KeyEventArgs e)
        {
            // Вставить в поле представления рисунка в виде HEX-массива строковое представление картинки из буфера обмена
            if (e.Control && !e.Shift && !e.Alt && e.KeyCode == Keys.V ||
                e.Shift && !e.Control && !e.Alt && e.KeyCode == Keys.Insert)
            {
                parseText(Clipboard.GetText());
                updateAll();
            }
        }

        private void labelPaint(object sender, PaintEventArgs e)
        {
            if (charWidth == 0 || charHeight == 0) return;

            var pnl_width = Convert.ToUInt32((pnlCharImage.ClientSize.Width - 2 * margin) / charWidth);
            var pnl_height = Convert.ToUInt32((pnlCharImage.ClientSize.Height - 2 * margin) / charHeight);

            if (pnl_width > 255) pnl_width = 255;
            if (pnl_height > 255) pnl_height = 255;

            // Размер "точки" изображения
            pnlSize = Convert.ToByte(Math.Min(pnl_width, pnl_height));

            // Левая позиция первого отрисовываемого Xex-значения
            pointLeftUp.X = Convert.ToInt16((pnlCharImage.ClientSize.Width - charWidth * pnlSize) / 2);

            Font font;

            if (pnlSize < 25)
                font = new Font("Consolas", 6.25F, FontStyle.Regular, GraphicsUnit.Point, 204);
            else if (pnlSize < 50)
                font = new Font("Consolas", 7.25F, FontStyle.Regular, GraphicsUnit.Point, 204);
            else if (pnlSize < 75)
                font = new Font("Consolas", 8.25F, FontStyle.Regular, GraphicsUnit.Point, 204);
            else
                font = new Font("Consolas", 9.75F, FontStyle.Regular, GraphicsUnit.Point, 204);

            e.Graphics.Clear(this.BackColor);
            e.Graphics.RotateTransform(-90);

            // Размер отрисовываемой строки
            var str = string.Format(charHeight <= 8 ? "0x{0:X2}" : (charHeight <= 16 ? "0x{0:X4}" : "0x{0:X8}"), 0);
            SizeF textSize = e.Graphics.MeasureString(str, font);

            var width = (int)textSize.Height + 2;
            var height = (int)textSize.Width + 2;

            for (var i = 0; i < charWidth; i++)
            {
                str = string.Format(charHeight <= 8 ? "0x{0:X2}" : (charHeight <= 16 ? "0x{0:X4}" : "0x{0:X8}"), Mask(uhex[i], charHeight));
                var left = pointLeftUp.X + i * pnlSize + (pnlSize - width) / 2 + 2;
                e.Graphics.DrawString(str, font, Brushes.Black, -height, left);                
            }
        }

        private void imagePaint(object sender, PaintEventArgs e)
        {
            if (charWidth == 0 || charHeight == 0) return;

            // Отрисовка текущего поля рисования
            var pnl_width = Convert.ToUInt32((pnlCharImage.ClientSize.Width - 2 * margin) / charWidth);
            var pnl_height = Convert.ToUInt32((pnlCharImage.ClientSize.Height - 2 * margin) / charHeight);
            
            if (pnl_width > 255) pnl_width = 255;
            if (pnl_height > 255) pnl_height = 255;

            // Размер "точки" изображения
            pnlSize = Convert.ToByte(Math.Min(pnl_width, pnl_height));

            // Левый верхний угол отрисовываемого изображения
            pointLeftUp.X = Convert.ToInt16((pnlCharImage.ClientSize.Width - charWidth * pnlSize) / 2);
            pointLeftUp.Y = Convert.ToInt16((pnlCharImage.ClientSize.Height - charHeight * pnlSize) / 2);

            var lineWidth = charWidth * pnlSize;
            var lineHeight = charHeight * pnlSize;

            float x1, x2, y1, y2;

            e.Graphics.Clear(Color.White);
            using (var pen = new Pen(Color.Black, 1.0F))
            using (var brush = new SolidBrush(Color.Black))
            {
                // Решетка
                y1 = pointLeftUp.Y;
                y2 = y1 + lineHeight;
                for (var i = 0; i <= charWidth; i++)
                {
                    x1 = x2 = pointLeftUp.X + i * pnlSize;
                    e.Graphics.DrawLine(pen, x1, y1, x2, y2);
                }
                x1 = pointLeftUp.X;
                x2 = x1 + lineWidth;
                for (var i = 0; i <= charHeight; i++)
                {
                    y1 = y2 = pointLeftUp.Y + i * pnlSize;
                    e.Graphics.DrawLine(pen, x1, y1, x2, y2);
                }
                // Заполненные точки
                for (var col = 0; col < charWidth; col++)
                {
                    uint val = uhex[col];
                    for (var row = 0; row < charHeight; row++)
                    {
                        var fill = (val & (0x01 << row)) > 0;
                        if (fill)
                        {
                            x1 = pointLeftUp.X + col * pnlSize;
                            y1 = pointLeftUp.Y + row * pnlSize;
                            e.Graphics.FillRectangle(brush, x1, y1, pnlSize, pnlSize);
                        }
                    }
                }
            }
        }

        private void updateAll()
        {
            // Необходимость перерисовать поле рисунка
            pnlCharImage.Invalidate();
            pnlHexLines.Invalidate();
            updateHexArray();
        }

        private void updateHexArray() 
        {
            // Обновить панель с текстовым представлением массива точек
            processing = true;
            edtHexArray.Text = generateTextFromPanel();
            processing = false;
        }

        private string generateTextFromPanel()
        {
            var charWidth = Convert.ToByte(edtFontWidth.Value);
            var charHeight = Convert.ToByte(edtFontHeight.Value);

            // Сформировать текстовое представления массива точек для вывода в нижнюю панель
            var div = charHeight <= 8 ? 16 : 8;
            var sb = new StringBuilder();
            sb.Append("{");
            for (var i = 0; i < charWidth; i++)
            {
                sb.Append(string.Format(charHeight <= 8 ? "0x{0:X2}" : (charHeight <= 16 ? "0x{0:X4}" : "0x{0:X8}"), Mask(uhex[i], charHeight)));
                if (i < charWidth - 1) sb.Append(", ");
                if ((i + 1) % div == 0 && i + 1 != charWidth) sb.Append("\r\n ");
            }
            sb.Append("}");
            return sb.ToString();
        }

        private void parseText(string text)
        {
            // Распарсить переданный в текстовой форме массив в массив точек uhex[]
            var parts = text.Trim(new[] { '{', '}' }).Replace(" ", "").Replace("\n", "").Replace("\r", "").Replace("\t", "").Replace(";", "").Split(',');
            for (var i = 0; i < Math.Min(32, parts.Length); i++)
            {
                try { uhex[i] = Convert.ToUInt32(parts[i], 16); } catch { }
            }
            pnlCharImage.Invalidate();
            pnlHexLines.Invalidate();
        }

        private uint Mask(uint val, byte len)
        {
            uint msk = 0;
            for (var i = 0; i < len; i++) msk |= (0x01U << i);
            return val & msk;
        }

        public void CheckWindowPosition(ref int x, ref int y, ref int w, ref int h)
        {
            Rectangle WorkingArea = Screen.GetWorkingArea(new Rectangle(x, y, w, h));
            if (x + w > WorkingArea.X + WorkingArea.Width) x = WorkingArea.X + WorkingArea.Width - w;
            if (y + h > WorkingArea.Y + WorkingArea.Height) y = WorkingArea.Y + WorkingArea.Height - h;
            if (x < WorkingArea.X) x = WorkingArea.X;
            if (y < WorkingArea.Y) y = WorkingArea.Y;
        }

        private void btnCopy_Click(object sender, EventArgs e)
        {
            Clipboard.SetText(edtHexArray.Text);
        }

        private void edtHexArray_TextChanged(object sender, EventArgs e)
        {
            if (processing) return;
            parseText(edtHexArray.Text);
        }

        private void edtHexArray_Leave(object sender, EventArgs e)
        {
            updateHexArray();
        }
    }
}
